<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi();

// PDO kullanarak veritabanı yedeği oluşturma fonksiyonu
// Bu fonksiyon artık şirket ID'si alacak ve sadece o şirkete ait verileri yedekleyecektir.
function generateDatabaseBackup($pdo, $database_name, $sirket_id = null) {
    try {
        $sql_dump = "-- PHP PDO Database Backup\n";
        $sql_dump .= "-- Generated on: " . date('Y-m-d H:i:s') . "\n";
        $sql_dump .= "-- Database: " . $database_name . "\n";
        if ($sirket_id !== null) {
            $sql_dump .= "-- Company ID: " . $sirket_id . " (Company-specific backup)\n";
        } else {
            $sql_dump .= "-- Type: Full System Backup\n";
        }
        $sql_dump .= "\n";
        
        $sql_dump .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
        $sql_dump .= "START TRANSACTION;\n";
        $sql_dump .= "SET time_zone = \"+00:00\";\n\n";
        
        // Yedeklenecek tabloların listesi. Buradaki her tablo için özel filtreleme yapılabilir.
        $tables_to_consider = [
            'ayarlar',
            'borclar',
            'cari_gruplari',
            'cariler',
            'hesap_hareketleri',
            'hesaplar',
            'kategoriler',
            'odeme_yontemleri',
            'stok_hareketleri',
            'teklif_urunleri',
            'teklifler',
            'urunler',
            'users', 
            'loglar', 
            'sirketler', 
            'kullanici_sirket_iliskisi', 
            'aktif_oturumlar', 
            'oturum_loglari' 
        ];

        // Veritabanındaki mevcut tabloları al
        $all_db_tables_stmt = $pdo->query("SHOW TABLES");
        $all_db_tables = $all_db_tables_stmt->fetchAll(PDO::FETCH_COLUMN);

        // Sadece veritabanında var olan ve yedeklenecek tablolar listesindeki tabloları işleme al
        $tables_to_backup_final = array_intersect($tables_to_consider, $all_db_tables);
        
        foreach ($tables_to_backup_final as $table) {
            // Tablo yapısını al
            $create_table = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(PDO::FETCH_ASSOC);
            $sql_dump .= "\n-- Table structure for table `$table`\n";
            $sql_dump .= "DROP TABLE IF EXISTS `$table`;\n";
            $sql_dump .= $create_table['Create Table'] . ";\n\n";
            
            $select_query = "";
            $params = [];
            $include_data = true; // Bu tablonun verilerinin yedeğe dahil edilip edilmeyeceği bayrağı

            if ($sirket_id !== null) { // Şirket bazlı yedekleme
                if ($table === 'sirketler') {
                    $select_query = "SELECT * FROM `sirketler` WHERE id = ?";
                    $params[] = $sirket_id;
                } elseif ($table === 'users') {
                    // Sadece bu şirkete bağlı kullanıcıları yedekle
                    $select_query = "SELECT u.* FROM `users` u JOIN `kullanici_sirket_iliskisi` ksi ON u.id = ksi.kullanici_id WHERE ksi.sirket_id = ?";
                    $params[] = $sirket_id;
                } elseif ($table === 'kullanici_sirket_iliskisi') {
                    // Sadece bu şirkete ait ilişkileri yedekle
                    $select_query = "SELECT * FROM `kullanici_sirket_iliskisi` WHERE sirket_id = ?";
                    $params[] = $sirket_id;
                } else {
                    // Diğer tablolarda sirket_id sütunu varsa filtrele, yoksa dahil etme
                    $has_sirket_id_column = false;
                    try {
                        $columns_stmt = $pdo->query("SHOW COLUMNS FROM `$table` LIKE 'sirket_id'");
                        $has_sirket_id_column = $columns_stmt->rowCount() > 0;
                    } catch (PDOException $e) {
                        // Sütun yoksa hata vermez
                    }

                    if ($has_sirket_id_column) {
                        $select_query = "SELECT * FROM `$table` WHERE sirket_id = ?";
                        $params[] = $sirket_id;
                    } else {
                        // Şirket ID'si olmayan ve özel olarak ele alınmayan tabloların verilerini dahil etme
                        $include_data = false; 
                    }
                }
            } else { // Tam sistem yedeği
                $select_query = "SELECT * FROM `$table`";
            }

            if ($include_data && !empty($select_query)) {
                $stmt = $pdo->prepare($select_query);
                $stmt->execute($params);
                $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (!empty($rows)) {
                    $sql_dump .= "-- Dumping data for table `$table`\n";
                    
                    foreach ($rows as $row) {
                        $columns = array_keys($row);
                        $values = array_values($row);
                        
                        // Değerleri escape et
                        $escaped_values = array_map(function($value) use ($pdo) {
                            if ($value === null) {
                                return 'NULL';
                            }
                            return $pdo->quote($value);
                        }, $values);
                        
                        $sql_dump .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $escaped_values) . ");\n";
                    }
                    $sql_dump .= "\n";
                }
            }
        }
        
        $sql_dump .= "COMMIT;\n";
        
        return $sql_dump;
        
    } catch (Exception $e) {
        error_log('Database backup error: ' . $e->getMessage());
        return false;
    }
}

// Yedekleme dizini
$backup_dir = __DIR__ . '/backups/';
if (!is_dir($backup_dir)) {
    mkdir($backup_dir, 0777, true);
}

$message = '';
$error = '';

// Kullanıcının erişim yetkisi olan şirketleri getir
$kullanici_sirketleri = [];
if (isset($_SESSION['kullanici_id'])) {
    $stmt_sirketler = $db->prepare("
        SELECT s.id, s.sirket_adi 
        FROM sirketler s
        JOIN kullanici_sirket_iliskisi ksi ON s.id = ksi.sirket_id
        WHERE ksi.kullanici_id = ?
        ORDER BY s.sirket_adi
    ");
    $stmt_sirketler->execute([$_SESSION['kullanici_id']]);
    $kullanici_sirketleri = $stmt_sirketler->fetchAll(PDO::FETCH_ASSOC);
}

if (isset($_POST['yedek_al'])) {
    // CSRF token kontrolü
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $error = "Geçersiz güvenlik tokenı!";
    } else {
        $selected_sirket_ids = $_POST['sirket_ids'] ?? []; // Seçilen şirket ID'leri dizisi
        
        if (empty($selected_sirket_ids)) {
            $error = 'Lütfen yedeklemek istediğiniz şirketleri seçin.';
        } else {
            // Kullanıcının gerçekten yetkili olduğu şirket ID'lerini filtrele
            $allowed_sirket_ids = array_column($kullanici_sirketleri, 'id');
            $valid_selected_sirket_ids = array_intersect($selected_sirket_ids, $allowed_sirket_ids);

            if (empty($valid_selected_sirket_ids)) {
                $error = 'Yedeklemek için yetkiniz olmayan şirketleri seçtiniz veya hiçbir geçerli şirket seçmediniz.';
            } else {
                $all_backups_successful = true;
                $successful_backups = [];
                $failed_backups = [];

                foreach ($valid_selected_sirket_ids as $sirket_id_to_backup) {
                    // Şirket adını dosya ismi için al
                    $sirket_adi_for_backup = 'genel'; // Varsayılan
                    foreach ($kullanici_sirketleri as $sirket) {
                        if ($sirket['id'] == $sirket_id_to_backup) {
                            $sirket_adi_for_backup = $sirket['sirket_adi'];
                            break;
                        }
                    }
                    
                    // Dosya adını şirket adı ile oluştur
                    $sanitized_sirket_adi = preg_replace('/[^a-zA-Z0-9_]/u', '', str_replace(' ', '_', mb_strtolower($sirket_adi_for_backup, 'UTF-8')));
                    $filename = 'Odjoo_yonetim_yedek_' . $sanitized_sirket_adi . '_' . date('Ymd_His') . '.sql';
                    $filepath = $backup_dir . $filename;

                    try {
                        // generateDatabaseBackup fonksiyonunu seçilen şirket ID'si ile çağır
                        $backup_content = generateDatabaseBackup($db, $dbname, $sirket_id_to_backup);
                        
                        if ($backup_content !== false) {
                            if (file_put_contents($filepath, $backup_content)) {
                                $successful_backups[] = $filename;
                            } else {
                                $failed_backups[] = 'Dosya yazma hatası: ' . $filename;
                                $all_backups_successful = false;
                            }
                        } else {
                            $failed_backups[] = 'Veritabanı yedeği alınamadı: ' . $sirket_adi_for_backup . ' (' . $sirket_id_to_backup . ')';
                            $all_backups_successful = false;
                        }
                    } catch (Exception $e) {
                        $failed_backups[] = 'Beklenmeyen hata (' . $sirket_adi_for_backup . '): ' . $e->getMessage();
                        error_log('Yedekleme exception: ' . $e->getMessage());
                        $all_backups_successful = false;
                    }
                }

                if ($all_backups_successful) {
                    $message = 'Tüm seçilen şirketlerin yedekleri başarıyla alındı: ' . implode(', ', $successful_backups);
                } else {
                    $message = 'Bazı yedeklemeler başarıyla tamamlandı: ' . implode(', ', $successful_backups) . ".";
                    $error = 'Bazı yedeklemeler başarısız oldu: ' . implode('; ', $failed_backups) . ". Yedekleme loglarını kontrol edin.";
                }
            }
        }
    }
}

// Mevcut yedekleri listele (sadece yetkili olunan şirketlere ait olanları göster)
$backups = [];
$allowed_sirket_ids_for_list = array_column($kullanici_sirketleri, 'id');

if (is_dir($backup_dir)) {
    $files = scandir($backup_dir);
    foreach ($files as $file) {
        if (pathinfo($file, PATHINFO_EXTENSION) == 'sql') {
            $sirket_id_from_filename = null;
            // Dosya adından şirket ID'sini veya adını çıkarmaya çalış
            // Örnek: Odjoo_yonetim_yedek_sirketadi_20231026_123456.sql
            if (preg_match('/^Odjoo_yonetim_yedek_([a-zA-Z0-9_]+)_(\d{8}_\d{6})\.sql$/', $file, $matches)) {
                $sirket_adi_in_filename = $matches[1];
                foreach ($kullanici_sirketleri as $sirket) {
                    $sanitized_sirket_adi = preg_replace('/[^a-zA-Z0-9_]/u', '', str_replace(' ', '_', mb_strtolower($sirket['sirket_adi'], 'UTF-8')));
                    if ($sanitized_sirket_adi === $sirket_adi_in_filename) {
                        $sirket_id_from_filename = $sirket['id'];
                        break;
                    }
                }
            }

            // Eğer dosya yetkili bir şirkete aitse veya genel yedekse listele
            if ($sirket_id_from_filename === null || in_array($sirket_id_from_filename, $allowed_sirket_ids_for_list)) {
                $backups[] = [
                    'name' => $file,
                    'size' => filesize($backup_dir . $file),
                    'date' => filemtime($backup_dir . $file),
                    'sirket_id' => $sirket_id_from_filename 
                ];
            }
        }
    }
    usort($backups, function($a, $b) { return $b['date'] - $a['date']; });
}

?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Yedek İşlemleri - Odjoo Yönetim</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.2/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; }
        .card { border: none; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.05); }
        .card-header { background-color: #007bff; color: white; border-radius: 10px 10px 0 0; }
        .btn-primary { background-color: #007bff; border-color: #007bff; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; }
        .table th, .table td { vertical-align: middle; }
        .form-check-input:checked {
            background-color: #007bff;
            border-color: #007bff;
        }
        .form-check-label strong {
            color: #333;
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container mt-4">
        <h2 class="mb-4"><i class="bi bi-database-fill-down me-2"></i>Sistem Yedekleme</h2>
        <hr>
        
        <?php if ($message): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill me-2"></i><?= $message ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle-fill me-2"></i><?= $error ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= csrf_token_olustur() ?>">

            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-building-fill-check me-2"></i>Yedeklenecek Şirketler</h5>
                </div>
                <div class="card-body">
                    <p class="card-text">Yedeklemek istediğiniz şirketleri seçin. Her seçilen şirket için ayrı bir veritabanı yedeği oluşturulacaktır.</p>
                    <?php if (!empty($kullanici_sirketleri)): ?>
                        <div class="row">
                            <?php foreach ($kullanici_sirketleri as $sirket): ?>
                                <div class="col-md-6 col-lg-4 mb-2">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="sirket_ids[]" value="<?= $sirket['id'] ?>" id="sirket_<?= $sirket['id'] ?>" checked>
                                        <label class="form-check-label" for="sirket_<?= $sirket['id'] ?>">
                                            <strong><?= htmlspecialchars($sirket['sirket_adi']) ?></strong>
                                        </label>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-circle me-2"></i>Erişim yetkiniz olan hiçbir şirket bulunamadı. Yedekleme işlemi yapılamaz.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <button type="submit" name="yedek_al" class="btn btn-primary btn-lg mt-3" <?= empty($kullanici_sirketleri) ? 'disabled' : '' ?>>
                <i class="bi bi-cloud-arrow-down me-2"></i>Seçilenleri Yedekle
            </button>
        </form>

        <h2 class="h5 mt-5 mb-3"><i class="bi bi-archive-fill me-2"></i>Mevcut Yedekler</h2>
        <?php if (empty($backups)): ?>
            <div class="alert alert-info" role="alert">
                <i class="bi bi-info-circle me-2"></i>Henüz hiç yedek alınmamış.
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-striped table-hover align-middle">
                    <thead>
                        <tr>
                            <th>Dosya Adı</th>
                            <th>Şirket</th>
                            <th>Boyut</th>
                            <th>Tarih</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($backups as $backup): ?>
                            <tr>
                                <td><?= temizle($backup['name']) ?></td>
                                <td>
                                    <?php 
                                        $sirket_adi_display = 'Genel/Bilinmiyor';
                                        if ($backup['sirket_id'] !== null) {
                                            foreach ($kullanici_sirketleri as $sirket_info) {
                                                if ($sirket_info['id'] == $backup['sirket_id']) {
                                                    $sirket_adi_display = htmlspecialchars($sirket_info['sirket_adi']);
                                                    break;
                                                }
                                            }
                                        }
                                        echo $sirket_adi_display;
                                    ?>
                                </td>
                                <td><?= round($backup['size'] / (1024 * 1024), 2) ?> MB</td>
                                <td><?= date('d.m.Y H:i:s', $backup['date']) ?></td>
                                <td>
                                    <?php 
                                        $can_access_backup = ($backup['sirket_id'] === null || in_array($backup['sirket_id'], $allowed_sirket_ids_for_list));
                                    ?>
                                    <a href="yedek_indir.php?file=<?= urlencode($backup['name']) ?>" 
                                       class="btn btn-success btn-sm me-2 <?= $can_access_backup ? '' : 'disabled' ?>" 
                                       <?= $can_access_backup ? '' : 'tabindex="-1" aria-disabled="true"' ?>
                                       title="<?= $can_access_backup ? 'Yedeği İndir' : 'Bu yedeğe erişim yetkiniz yok' ?>">
                                        <i class="bi bi-download"></i> İndir
                                    </a>
                                    <a href="yedek_sil.php?file=<?= urlencode($backup['name']) ?>" 
                                       class="btn btn-danger btn-sm <?= $can_access_backup ? '' : 'disabled' ?>" 
                                       onclick="<?= $can_access_backup ? 'return confirm(\'Bu yedeği silmek istediğinizden emin misiniz?\');' : '' ?>"
                                       <?= $can_access_backup ? '' : 'tabindex="-1" aria-disabled="true"' ?>
                                       title="<?= $can_access_backup ? 'Yedeği Sil' : 'Bu yedeği silme yetkiniz yok' ?>">
                                        <i class="bi bi-trash"></i> Sil
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

