<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../functions/stok_fonksiyonlari.php';
yetkili_mi();

$sayfa_baslik = "Ürün Sil";

// Ürün ID'sini al
$urun_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$urun_id) {
    $_SESSION['hata'] = "Geçersiz ürün ID'si.";
    header("Location: urunler.php");
    exit();
}

$paraBirimi = getParaBirimiSembolu();

// Ürün bilgilerini getir
try {
    $stmt = $db->prepare("
        SELECT u.*, ms.miktar as stok_miktari, ms.ortalama_maliyet, k.kategori_adi
        FROM urunler u
        LEFT JOIN mevcut_stok ms ON u.id = ms.urun_id AND ms.sirket_id = u.sirket_id
        LEFT JOIN kategoriler k ON u.kategori_id = k.id AND k.sirket_id = u.sirket_id
        WHERE u.id = ? AND u.sirket_id = ? AND u.aktif = 1
    ");
    $stmt->execute([$urun_id, $_SESSION['aktif_sirket_id']]);
    $urun = $stmt->fetch();
    
    if (!$urun) {
        $_SESSION['hata'] = "Ürün bulunamadı veya bu şirkete ait değil.";
        header("Location: urunler.php");
        exit();
    }
} catch (PDOException $e) {
    error_log("Ürün getirme hatası: " . $e->getMessage());
    $_SESSION['hata'] = "Ürün bilgileri yüklenirken bir hata oluştu.";
    header("Location: urunler.php");
    exit();
}

// Ürünün kullanım durumunu kontrol et
$kullanim_kontrolleri = [];
try {
    // Fatura detaylarında kullanılıyor mu?
    $stmt = $db->prepare("SELECT COUNT(*) as adet FROM fatura_detaylari WHERE urun_id = ? AND sirket_id = ?");
    $stmt->execute([$urun_id, $_SESSION['aktif_sirket_id']]);
    $fatura_kullanim = $stmt->fetch()['adet'];
    if ($fatura_kullanim > 0) {
        $kullanim_kontrolleri[] = "Bu ürün {$fatura_kullanim} adet fatura detayında kullanılmış.";
    }
    
    // Stok hareketleri var mı?
    $stmt = $db->prepare("SELECT COUNT(*) as adet FROM stok_hareketleri WHERE urun_id = ? AND sirket_id = ?");
    $stmt->execute([$urun_id, $_SESSION['aktif_sirket_id']]);
    $stok_hareket_sayisi = $stmt->fetch()['adet'];
    if ($stok_hareket_sayisi > 0) {
        $kullanim_kontrolleri[] = "Bu ürün için {$stok_hareket_sayisi} adet stok hareketi kaydı bulunuyor.";
    }
    
    // Teklif detaylarında kullanılıyor mu?
    $stmt = $db->prepare("SELECT COUNT(*) as adet FROM teklif_detaylari WHERE urun_id = ? AND sirket_id = ?");
    $stmt->execute([$urun_id, $_SESSION['aktif_sirket_id']]);
    $teklif_kullanim = $stmt->fetch()['adet'];
    if ($teklif_kullanim > 0) {
        $kullanim_kontrolleri[] = "Bu ürün {$teklif_kullanim} adet teklif detayında kullanılmış.";
    }
    
} catch (PDOException $e) {
    error_log("Kullanım kontrolü hatası: " . $e->getMessage());
    $kullanim_kontrolleri[] = "Kullanım kontrolü yapılamadı.";
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
        header("Location: urun_sil.php?id=" . $urun_id);
        exit();
    }

    $silme_onay = $_POST['silme_onay'] ?? '';
    if ($silme_onay !== 'EVET') {
        $_SESSION['hata'] = "Silme işlemi için 'EVET' yazmanız gerekiyor.";
        header("Location: urun_sil.php?id=" . $urun_id);
        exit();
    }

    try {
        $db->beginTransaction();
        
        // Ürünü pasif yap (soft delete)
        $stmt = $db->prepare("UPDATE urunler SET aktif = 0, guncelleme_tarihi = NOW() WHERE id = ? AND sirket_id = ?");
        $stmt->execute([$urun_id, $_SESSION['aktif_sirket_id']]);
        
        // Log ekle
        sistem_log_ekle($db, 'sil', 'urunler', $urun_id, $_SESSION['aktif_sirket_id'], $urun, null, 'Ürün silindi: ' . $urun['urun_adi']);
        
        $db->commit();
        
        $_SESSION['mesaj'] = "Ürün başarıyla silindi.";
        header("Location: urunler.php");
        exit();
        
    } catch (Exception $e) {
        $db->rollBack();
        error_log("Ürün silme hatası: " . $e->getMessage());
        $_SESSION['hata'] = "Ürün silinirken bir hata oluştu.";
    }
}

$csrf_token = csrf_token_olustur();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo Yönetim - <?= temizle($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container mt-4">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bi bi-trash text-danger"></i> <?= temizle($sayfa_baslik) ?></h2>
                    <div>
                        <a href="urun_duzenle.php?id=<?= $urun['id'] ?>" class="btn btn-outline-primary me-2">
                            <i class="bi bi-pencil"></i> Düzenle
                        </a>
                        <a href="urunler.php" class="btn btn-outline-secondary">
                            <i class="bi bi-arrow-left"></i> Ürün Listesi
                        </a>
                    </div>
                </div>

                <!-- Uyarı Mesajı -->
                <div class="alert alert-danger" role="alert">
                    <h4 class="alert-heading"><i class="bi bi-exclamation-triangle-fill me-2"></i>Dikkat!</h4>
                    <p>Bu ürünü silmek üzeresiniz. Bu işlem geri alınamaz!</p>
                    <hr>
                    <p class="mb-0">Ürün silindiğinde:</p>
                    <ul class="mb-0">
                        <li>Ürün listesinden kaldırılacak</li>
                        <li>Yeni satış/alış işlemlerinde kullanılamayacak</li>
                        <li>Mevcut fatura ve stok hareketleri korunacak</li>
                    </ul>
                </div>

                <!-- Kullanım Durumu -->
                <?php if (!empty($kullanim_kontrolleri)): ?>
                    <div class="alert alert-warning" role="alert">
                        <h5><i class="bi bi-info-circle me-2"></i>Kullanım Durumu</h5>
                        <ul class="mb-0">
                            <?php foreach ($kullanim_kontrolleri as $kontrol): ?>
                                <li><?= htmlspecialchars($kontrol) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <!-- Ürün Bilgileri -->
                <div class="card mb-4">
                    <div class="card-header bg-light">
                        <h5 class="mb-0"><i class="bi bi-box-seam me-2"></i>Silinecek Ürün Bilgileri</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <td><strong>Ürün Adı:</strong></td>
                                        <td><?= temizle($urun['urun_adi']) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Barkod:</strong></td>
                                        <td><?= !empty($urun['barkod']) ? temizle($urun['barkod']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Kategori:</strong></td>
                                        <td><?= temizle($urun['kategori_adi'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Birim:</strong></td>
                                        <td><?= temizle($urun['birim']) ?></td>
                                    </tr>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <td><strong>Alış Fiyatı:</strong></td>
                                        <td><?= number_format($urun['alis_fiyati'], 2, ',', '.') ?> <?= $paraBirimi ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Satış Fiyatı:</strong></td>
                                        <td><?= number_format($urun['satis_fiyati'], 2, ',', '.') ?> <?= $paraBirimi ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>KDV Oranı:</strong></td>
                                        <td>%<?= number_format($urun['kdv_orani'], 0) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Mevcut Stok:</strong></td>
                                        <td>
                                            <span class="badge bg-primary"><?= number_format($urun['stok_miktari'] ?? 0, 2, ',', '.') ?> <?= temizle($urun['birim']) ?></span>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <?php if (!empty($urun['aciklama'])): ?>
                            <div class="row">
                                <div class="col-12">
                                    <strong>Açıklama:</strong><br>
                                    <p class="text-muted"><?= nl2br(temizle($urun['aciklama'])) ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="row">
                            <div class="col-12">
                                <small class="text-muted">
                                    <strong>Oluşturulma Tarihi:</strong> <?= date('d.m.Y H:i', strtotime($urun['olusturulma_tarihi'])) ?>
                                </small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Silme Formu -->
                <div class="card border-danger">
                    <div class="card-header bg-danger text-white">
                        <h5 class="mb-0"><i class="bi bi-trash me-2"></i>Silme Onayı</h5>
                    </div>
                    <div class="card-body">
                        <form method="post" id="silmeForm">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            
                            <div class="mb-3">
                                <label for="silme_onay" class="form-label">
                                    Bu ürünü silmek istediğinizi onaylamak için <strong>"EVET"</strong> yazın:
                                </label>
                                <input type="text" class="form-control" id="silme_onay" name="silme_onay" 
                                       placeholder="EVET yazın..." required autocomplete="off">
                            </div>
                            
                            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                <a href="urunler.php" class="btn btn-secondary me-md-2">
                                    <i class="bi bi-x-circle me-2"></i> İptal
                                </a>
                                <button type="submit" class="btn btn-danger" id="silBtn" disabled>
                                    <i class="bi bi-trash me-2"></i> Ürünü Sil
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener("DOMContentLoaded", function() {
            const silmeOnayInput = document.getElementById("silme_onay");
            const silBtn = document.getElementById("silBtn");
            
            silmeOnayInput.addEventListener("input", function() {
                if (this.value.trim().toUpperCase() === "EVET") {
                    silBtn.disabled = false;
                    silBtn.classList.remove("btn-secondary");
                    silBtn.classList.add("btn-danger");
                } else {
                    silBtn.disabled = true;
                    silBtn.classList.remove("btn-danger");
                    silBtn.classList.add("btn-secondary");
                }
            });
            
            // Form gönderilmeden önce son onay
            document.getElementById("silmeForm").addEventListener("submit", function(e) {
                if (!confirm("Bu ürünü silmek istediğinizden kesinlikle emin misiniz?\n\nBu işlem geri alınamaz!")) {
                    e.preventDefault();
                }
            });
        });
    </script>
    
    <?php include __DIR__ . '/../includes/footer.php'; ?>
</body>
</html>

