<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi();

$sayfa_baslik = "Toplu İşlemler";
$paraBirimi = getParaBirimiSembolu(); // Para birimini al

// İşlem türü
$islem_turu = isset($_GET['tur']) ? $_GET['tur'] : 'odeme';
$aktif_sirket_id = $_SESSION['aktif_sirket_id'];

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
        header("Location: toplu_islem.php?tur=" . $islem_turu);
        exit();
    }
    
    // Form verilerini al
    $secilen_cariler = isset($_POST['cari_ids']) ? $_POST['cari_ids'] : [];
    $hesap_id = filter_input(INPUT_POST, 'hesap_id', FILTER_VALIDATE_INT);
    $tutar_str = str_replace(',', '.', $_POST['tutar'] ?? '0');
    $tutar = filter_var($tutar_str, FILTER_VALIDATE_FLOAT);
    $aciklama = temizle($_POST['aciklama'] ?? '');
    $tarih = filter_input(INPUT_POST, 'tarih', FILTER_SANITIZE_SPECIAL_CHARS);
    
    // Veri doğrulama
    $hatalar = [];
    if (empty($secilen_cariler)) $hatalar[] = "En az bir cari seçmelisiniz.";
    if (empty($hesap_id)) $hatalar[] = "Hesap seçimi zorunludur.";
    if ($tutar === false || $tutar <= 0) $hatalar[] = "Geçerli bir tutar giriniz (0'dan büyük).";
    if (empty($tarih) || !preg_match("/^\d{4}-\d{2}-\d{2}$/", $tarih)) $hatalar[] = "Geçerli bir işlem tarihi giriniz (YYYY-AA-GG).";
    
    // Hesap bakiyesi kontrolü (Toplu ödeme için)
    if ($islem_turu == 'odeme' && !empty($hesap_id)) {
        $stmt_bakiye = $db->prepare("SELECT bakiye FROM hesaplar WHERE id = ? AND sirket_id = ?");
        $stmt_bakiye->execute([$hesap_id, $aktif_sirket_id]);
        $hesap_bakiye = $stmt_bakiye->fetchColumn();
        
        $toplam_odeme = $tutar * count($secilen_cariler);
        if ($hesap_bakiye === false || $hesap_bakiye < $toplam_odeme) {
            $hatalar[] = "Hesapta yeterli bakiye yok! Toplam gerekli: ".number_format($toplam_odeme, 2, ',', '.')." $paraBirimi, Mevcut: ".number_format($hesap_bakiye ?: 0, 2, ',', '.')." $paraBirimi";
        }
    }
    
    if (empty($hatalar)) {
        try {
            $db->beginTransaction();
            $basarili_islemler = 0;
            $hatali_islemler = [];
            
            foreach ($secilen_cariler as $cari_id) {
                try {
                    if ($islem_turu == 'odeme') {
                        // Toplu ödeme (bizden tedarikçilere)
                        
                        // 1. Hesap bakiyesini düşür
                        if (!hesapBakiyeGuncelle($db, $hesap_id, -$tutar)) { // Negatif değer
                            throw new Exception("Hesap bakiyesi güncellenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        // 2. Hesap hareketi ekle ('cikis' olarak)
                        $hareket_aciklama = "Toplu Ödeme: " . $aciklama;
                        if (!hesapHareketiEkle($db, $hesap_id, $aktif_sirket_id, 'cikis', $tutar, $hareket_aciklama, $cari_id, null)) {
                            throw new Exception("Hesap hareketi eklenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        // 3. Cari'ye olan borcumuzu azalt (toplam_alacak azalır)
                        if (!cariBakiyeGuncelle($db, $cari_id, 0.0, -$tutar)) { // Alacak azalıyor
                            throw new Exception("Cari bakiyesi güncellenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        $basarili_islemler++;
                    } else {
                        // Toplu tahsilat (müşterilerden bize)
                        
                        // 1. Hesap bakiyesini artır
                        if (!hesapBakiyeGuncelle($db, $hesap_id, $tutar)) {
                            throw new Exception("Hesap bakiyesi güncellenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        // 2. Hesap hareketi ekle ('giris' olarak)
                        $hareket_aciklama = "Toplu Tahsilat: " . $aciklama;
                        if (!hesapHareketiEkle($db, $hesap_id, $aktif_sirket_id, 'giris', $tutar, $hareket_aciklama, $cari_id, null)) {
                            throw new Exception("Hesap hareketi eklenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        // 3. Cari'nin borcunu azalt (toplam_borc azalır)
                        if (!cariBakiyeGuncelle($db, $cari_id, -$tutar, 0.0)) { // Borç azalıyor
                            throw new Exception("Cari bakiyesi güncellenemedi (Cari ID: {$cari_id}).");
                        }
                        
                        $basarili_islemler++;
                    }
                } catch (Exception $e) {
                    $hatali_islemler[] = "Cari ID {$cari_id}: " . $e->getMessage();
                }
            }
            
            if (count($hatali_islemler) > 0) {
                $db->rollBack();
                $_SESSION['hata'] = "Bazı işlemler başarısız oldu: " . implode(", ", $hatali_islemler);
            } else {
                $db->commit();
                $_SESSION['mesaj'] = "Toplu işlem başarıyla tamamlandı. Toplam {$basarili_islemler} işlem gerçekleştirildi.";
            }
            
            header("Location: toplu_islem.php?tur=" . $islem_turu);
            exit();
            
        } catch (Exception $e) {
            $db->rollBack();
            error_log("Toplu işlem hatası: " . $e->getMessage());
            $_SESSION['hata'] = "İşlem sırasında bir hata oluştu: " . $e->getMessage();
            $_SESSION['form_verileri_toplu'] = $_POST;
            header("Location: toplu_islem.php?tur=" . $islem_turu);
            exit();
        }
    } else {
        $_SESSION['form_hatalari_toplu'] = $hatalar;
        $_SESSION['form_verileri_toplu'] = $_POST;
        header("Location: toplu_islem.php?tur=" . $islem_turu);
        exit();
    }
}

// Carileri ve hesapları getir
try {
    // Cari listesi (işlem türüne göre filtrele)
    if ($islem_turu == 'odeme') {
        // Tedarikçiler (bizim borçlu olduklarımız)
        $cariler = $db->query("
            SELECT id, ad_soyad, toplam_alacak, bakiye
            FROM cariler 
            WHERE toplam_alacak > 0 AND sirket_id = $aktif_sirket_id
            ORDER BY ad_soyad
        ")->fetchAll(PDO::FETCH_ASSOC);
    } else {
        // Müşteriler (bize borçlu olanlar)
        $cariler = $db->query("
            SELECT id, ad_soyad, toplam_borc, bakiye
            FROM cariler 
            WHERE toplam_borc > 0 AND sirket_id = $aktif_sirket_id
            ORDER BY ad_soyad
        ")->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Hesaplar
    $hesaplar = $db->query("
        SELECT id, hesap_adi, bakiye 
        FROM hesaplar 
        WHERE sirket_id = $aktif_sirket_id
        ORDER BY hesap_adi
    ")->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Toplu işlem veri çekme hatası: " . $e->getMessage());
    $_SESSION['hata'] = "Veriler yüklenirken bir hata oluştu.";
    $cariler = [];
    $hesaplar = [];
}

$csrf_token = csrf_token_olustur();
$form_hatalari = $_SESSION['form_hatalari_toplu'] ?? [];
$form_verileri = $_SESSION['form_verileri_toplu'] ?? [];
unset($_SESSION['form_hatalari_toplu'], $_SESSION['form_verileri_toplu']);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo Yönetim - <?= temizle($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        .cari-list {
            max-height: 400px;
            overflow-y: auto;
        }
        .cari-item {
            padding: 8px;
            border-bottom: 1px solid #eee;
            transition: background-color 0.2s;
        }
        .cari-item:hover {
            background-color: #f8f9fa;
        }
        .cari-item label {
            cursor: pointer;
            width: 100%;
            margin-bottom: 0;
        }
        .badge-borc { background-color: #dc3545; }
        .badge-alacak { background-color: #198754; }
        
        /* Mobil uyumluluk */
        @media (max-width: 768px) {
            .cari-list {
                max-height: 300px;
            }
            .cari-item {
                padding: 10px 5px;
            }
            .card-title {
                font-size: 1.1rem;
            }
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h2><i class="bi bi-list-check"></i> <?= temizle($sayfa_baslik) ?></h2>
            <div class="btn-group">
                <a href="?tur=tahsilat" class="btn btn-<?= $islem_turu == 'tahsilat' ? 'success' : 'outline-success' ?>">
                    <i class="bi bi-arrow-down-circle"></i> Toplu Tahsilat
                </a>
                <a href="?tur=odeme" class="btn btn-<?= $islem_turu == 'odeme' ? 'danger' : 'outline-danger' ?>">
                    <i class="bi bi-arrow-up-circle"></i> Toplu Ödeme
                </a>
            </div>
        </div>
        <hr>
        
        <?php if (!empty($form_hatalari)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <strong><i class="bi bi-exclamation-triangle-fill me-2"></i> Lütfen hataları düzeltin:</strong>
                <ul>
                    <?php foreach ($form_hatalari as $hata): ?>
                        <li><?= htmlspecialchars($hata) ?></li>
                    <?php endforeach; ?>
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        
        <?php if (empty($cariler)): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle-fill me-2"></i> 
                <?= $islem_turu == 'odeme' ? 'Ödeme yapılacak tedarikçi bulunamadı.' : 'Tahsilat yapılacak müşteri bulunamadı.' ?>
            </div>
        <?php else: ?>
            <form method="post" id="topluIslemForm">
                <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                
                <div class="row">
                    <div class="col-md-7">
                        <div class="card shadow-sm mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="bi bi-people-fill me-2"></i>
                                    <?= $islem_turu == 'odeme' ? 'Tedarikçi Seçimi' : 'Müşteri Seçimi' ?>
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <input type="text" id="cariArama" class="form-control" placeholder="Cari ara...">
                                </div>
                                <div class="border rounded p-2 cari-list">
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="tumunuSec">
                                        <label class="form-check-label" for="tumunuSec">
                                            <strong>Tümünü Seç</strong>
                                        </label>
                                    </div>
                                    <hr class="my-1">
                                    <?php foreach ($cariler as $cari): ?>
                                    <div class="form-check cari-item" data-cari-adi="<?= strtolower(htmlspecialchars($cari['ad_soyad'])) ?>">
                                        <input class="form-check-input" type="checkbox" name="cari_ids[]" value="<?= $cari['id'] ?>" id="cari_<?= $cari['id'] ?>">
                                        <label class="form-check-label d-flex justify-content-between" for="cari_<?= $cari['id'] ?>">
                                            <span><?= htmlspecialchars($cari['ad_soyad']) ?></span>
                                            <?php if ($islem_turu == 'odeme'): ?>
                                                <span class="badge text-bg-danger">Borç: <?= number_format($cari['toplam_alacak'], 2, ',', '.') ?> <?= $paraBirimi ?></span>
                                            <?php else: ?>
                                                <span class="badge text-bg-success">Alacak: <?= number_format($cari['toplam_borc'], 2, ',', '.') ?> <?= $paraBirimi ?></span>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-5">
                        <div class="card shadow-sm">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="bi bi-cash-coin me-2"></i>
                                    <?= $islem_turu == 'odeme' ? 'Ödeme Bilgileri' : 'Tahsilat Bilgileri' ?>
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <label for="hesap_id" class="form-label">Hesap *</label>
                                    <select class="form-select" id="hesap_id" name="hesap_id" required>
                                        <option value="">Hesap Seçin</option>
                                        <?php foreach ($hesaplar as $hesap): ?>
                                            <option value="<?= $hesap['id'] ?>" <?= (isset($form_verileri['hesap_id']) && $form_verileri['hesap_id'] == $hesap['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($hesap['hesap_adi']) ?> (<?= number_format($hesap['bakiye'], 2, ',', '.') ?> <?= $paraBirimi ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="tutar" class="form-label">Cari Başına Tutar *</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="tutar" name="tutar" value="<?= htmlspecialchars($form_verileri['tutar'] ?? '') ?>" required>
                                        <span class="input-group-text"><?= $paraBirimi ?></span>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="tarih" class="form-label">İşlem Tarihi *</label>
                                    <input type="date" class="form-control" id="tarih" name="tarih" value="<?= htmlspecialchars($form_verileri['tarih'] ?? date('Y-m-d')) ?>" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="aciklama" class="form-label">Açıklama</label>
                                    <textarea class="form-control" id="aciklama" name="aciklama" rows="3"><?= htmlspecialchars($form_verileri['aciklama'] ?? '') ?></textarea>
                                </div>
                                
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="bi bi-check-circle-fill me-2"></i> İşlemi Tamamla
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        <?php endif; ?>
    </div>

    <?php include __DIR__ . '/../includes/footer.php'; ?>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const tumunuSec = document.getElementById('tumunuSec');
        const cariListesi = document.querySelectorAll('.cari-item input[type="checkbox"]');
        const cariArama = document.getElementById('cariArama');

        if (tumunuSec) {
            tumunuSec.addEventListener('change', function() {
                cariListesi.forEach(checkbox => {
                    const cariItem = checkbox.closest('.cari-item');
                    if (window.getComputedStyle(cariItem).display !== 'none') {
                        checkbox.checked = tumunuSec.checked;
                    }
                });
            });
        }

        if (cariArama) {
            cariArama.addEventListener('input', function() {
                const aramaTerimi = this.value.toLowerCase();
                document.querySelectorAll('.cari-item').forEach(item => {
                    const cariAdi = item.dataset.cariAdi;
                    if (cariAdi.includes(aramaTerimi)) {
                        item.style.display = 'flex';
                    } else {
                        item.style.display = 'none';
                    }
                });
            });
        }
    });
    </script>
</body>
</html>
