<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/language_functions.php';
yetkili_mi('teklifler');

// Initialize language
$current_language = getCurrentLanguage();
$translation_file = __DIR__ . "/../languages/{$current_language}.php";
$translations = file_exists($translation_file) ? require $translation_file : [];

// Translation function
if (!function_exists('__')) {
    function __($key, $params = []) {
        global $translations;
        $translation = $translations[$key] ?? $key;

        // Replace placeholders if any
        if (!empty($params)) {
            foreach ($params as $param => $value) {
                $translation = str_replace(':' . $param, $value, $translation);
            }
        }

        return $translation;
    }
}

// tFPDF kütüphanesini dahil et
require_once __DIR__ . '/tfpdf/tfpdf.php';

// Teklif ID'sini al
$teklif_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$teklif_id) {
    die(__("invalid_quote_id"));
}

$aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;
$teklif = null;
$teklif_detaylari = [];
$sirket_bilgileri = null;
$cari = null;

try {
    // Döviz kurları
    $stmt_doviz = $db->prepare("SELECT para_birimi, satis FROM doviz_kurlari");
    $stmt_doviz->execute();
    $doviz_kurlari = [];
    while ($row = $stmt_doviz->fetch(PDO::FETCH_ASSOC)) {
        $doviz_kurlari[$row['para_birimi']] = $row['satis'];
    }
    
    // Şirket Bilgileri
    $stmt_sirket = $db->prepare("SELECT * FROM sirketler WHERE id = ?");
    $stmt_sirket->execute([$aktif_sirket_id]);
    $sirket_bilgileri = $stmt_sirket->fetch(PDO::FETCH_ASSOC);

    // Teklif ana bilgileri
    $stmt_teklif = $db->prepare("SELECT *, COALESCE(tutar, 0) as tutar FROM teklifler WHERE id = ? AND sirket_id = ?");
    $stmt_teklif->execute([$teklif_id, $aktif_sirket_id]);
    $teklif = $stmt_teklif->fetch(PDO::FETCH_ASSOC);

    if (!$teklif) {
        die(__("quote_not_found_or_not_belongs"));
    }

    // Cari bilgileri
    $stmt_cari = $db->prepare("SELECT * FROM cariler WHERE id = ? AND sirket_id = ?");
    $stmt_cari->execute([$teklif['cari_id'], $aktif_sirket_id]);
    $cari = $stmt_cari->fetch(PDO::FETCH_ASSOC);

    // Teklif detayları (ürünler)
    $stmt_detaylar = $db->prepare("
        SELECT td.*, u.urun_adi, u.birim, u.aciklama as urun_aciklama, u.alis_para_birimi, u.satis_para_birimi
        FROM teklif_detaylari td 
        JOIN urunler u ON td.urun_id = u.id 
        WHERE td.teklif_id = ?
        ORDER BY td.id
    ");
    $stmt_detaylar->execute([$teklif_id]);
    $teklif_detaylari = $stmt_detaylar->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    die("Veritabanı hatası: " . $e->getMessage());
}

// PDF Sınıfı - Güzel tasarım için
class PDF_Teklif extends tFPDF
{
    public $sirket_bilgileri;
    public $teklif;
    public $cari;

    function Header()
    {
        // Sayfa üstü mavi çizgi
        $this->SetFillColor(13, 110, 253);
        $this->Rect(0, 0, 210, 3, 'F');
        
        // Logo (varsa)
        if (!empty($this->sirket_bilgileri['logo_yolu']) && file_exists(__DIR__ . '/../' . $this->sirket_bilgileri['logo_yolu'])) {
            $this->Image(__DIR__ . '/../' . $this->sirket_bilgileri['logo_yolu'], 15, 15, 25);
        }
        
        // Sol taraf - Şirket bilgileri
        $this->SetFont('DejaVu', 'B', 14);
        $this->SetTextColor(33, 37, 41);
        $this->Cell(0, 20, '', 0, 1);
        $this->Cell(95, 8, $this->sirket_bilgileri['sirket_adi'] ?? '', 0, 0);
        
        // Sağ taraf - TEKLİF başlığı
        $this->SetFont('DejaVu', 'B', 24);
        $this->SetTextColor(13, 110, 253);
        $this->Cell(95, 8, strtoupper(__('quote')), 0, 1, 'R');
        
        // Sol taraf - Şirket adres
        $this->SetFont('DejaVu', '', 10);
        $this->SetTextColor(100, 100, 100);
        if (!empty($this->sirket_bilgileri['adres'])) {
            $this->Cell(95, 6, $this->sirket_bilgileri['adres'], 0, 0);
        }
        
        // Sağ taraf - Teklif bilgileri
        $this->SetFont('DejaVu', '', 10);
        $this->Cell(95, 6, __('quote_number') . ': #' . str_pad($this->teklif['id'], 6, '0', STR_PAD_LEFT), 0, 1, 'R');
        
        // Sol taraf - Şirket telefon (varsa)
        if (!empty($this->sirket_bilgileri['telefon'])) {
            $this->SetFont('DejaVu', '', 10);
            $this->SetTextColor(100, 100, 100);
            $this->Cell(95, 6, 'Tel: ' . $this->sirket_bilgileri['telefon'], 0, 0);
        }
        
        // Sağ taraf - Tarih
        $this->SetFont('DejaVu', '', 10);
        $this->Cell(95, 6, __('date') . ': ' . date('d.m.Y', strtotime($this->teklif['teklif_tarihi'])), 0, 1, 'R');
        
        // Sağ taraf - Geçerlilik tarihi (varsa)
        if (!empty($this->teklif['gecerlilik_tarihi'])) {
            $this->Cell(95, 6, '', 0, 0); // left spacer to align to right column
            $this->Cell(95, 6, __('valid_until') . ': ' . date('d.m.Y', strtotime($this->teklif['gecerlilik_tarihi'])), 0, 1, 'R');
        }
        
        $this->Ln(10);
    }

    function Footer()
    {
        $this->SetY(-25);
        
        // Üst çizgi
        $this->SetDrawColor(222, 226, 230);
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        
        // Alt bilgiler
        $this->SetFont('DejaVu', '', 8);
        $this->SetTextColor(100, 100, 100);
        
        $website = !empty($this->sirket_bilgileri['web_sitesi']) ? $this->sirket_bilgileri['web_sitesi'] : '';
        $footer_text = str_replace(':website', $website, __('generated_by_odjoo'));
        
        $this->Cell(0, 5, $footer_text, 0, 1, 'C');
        
        // Sayfa numarası
        $this->Cell(0, 5, str_replace([':current', ':total'], [$this->PageNo(), '{nb}'], __('page_x_of_y')), 0, 0, 'C');
    }
}

// PDF oluştur
$pdf = new PDF_Teklif();
$pdf->sirket_bilgileri = $sirket_bilgileri;
$pdf->teklif = $teklif;
$pdf->cari = $cari;

$pdf->AddFont('DejaVu', '', 'DejaVuSans.ttf', true);
$pdf->AddFont('DejaVu', 'B', 'DejaVuSans-Bold.ttf', true);
$pdf->AliasNbPages();
    $pdf->AddPage();

// Ana içerik - Sadece müşteri bilgileri (tek satır, sağa yaslı)
// Hesaplamalar: teklif toplamı ve geçerlilik gün sayısı
$pdf->SetFont('DejaVu', '', 10);
$pdf->SetTextColor(100, 100, 100);
$ara_toplam = 0;
$toplam_kdv = 0;
$genel_toplam = 0;
$gecerlilik_gun = 0;
if (!empty($teklif['gecerlilik_tarihi'])) {
    $teklif_tarihi = new DateTime($teklif['teklif_tarihi']);
    $gecerlilik_tarihi = new DateTime($teklif['gecerlilik_tarihi']);
    $gecerlilik_gun = $teklif_tarihi->diff($gecerlilik_tarihi)->days;
}

$pdf->Ln(10);

// Para birimi notu kaldırıldı

// Ürün Tablosu - Güzel tasarım
$pdf->SetFont('DejaVu', 'B', 10);
$pdf->SetFillColor(248, 249, 250);
$pdf->SetDrawColor(222, 226, 230);
$pdf->SetTextColor(33, 37, 41);

// Teklifin stoklu mu stoksuz mu olduğunu belirle
$is_stoksuz = empty($teklif_detaylari) && !empty($teklif['aciklama']);

// Tablo başlığı
$pdf->SetFont('DejaVu', 'B', 9);
$pdf->SetFillColor(240, 240, 240);
$pdf->SetTextColor(0);

if ($is_stoksuz) {
    $pdf->Cell(15, 10, '#', 1, 0, 'C', true);
    $pdf->Cell(143, 10, __('description'), 1, 0, 'C', true);
    $pdf->Cell(30, 10, __('amount'), 1, 1, 'C', true);
} else {
    $pdf->Cell(15, 10, '#', 1, 0, 'C', true);
    $pdf->Cell(75, 10, __('product_service_name'), 1, 0, 'C', true);
    $pdf->Cell(22, 10, __('quantity'), 1, 0, 'C', true);
    $pdf->Cell(28, 10, __('unit_price'), 1, 0, 'C', true);
    $pdf->Cell(18, 10, __('vat') . ' %', 1, 0, 'C', true);
    $pdf->Cell(30, 10, __('amount'), 1, 1, 'C', true);
}

// Tablo içeriği
$pdf->SetFont('DejaVu', '', 8);
$ara_toplam = 0;
$toplam_kdv = $teklif['toplam_kdv'] ?? 0;

if ($is_stoksuz) {
    $satirlar = preg_split('/\r\n|\r|\n/', trim($teklif['aciklama']));
    $num = 1;
    foreach ($satirlar as $satir) {
        if (trim($satir) === '') continue;
        if (preg_match('/^(.*) \((.*)\)$/', $satir, $matches)) {
            $aciklama = trim($matches[1]);
            $tutar_str = trim($matches[2]);
            $pdf->Cell(15, 7, $num++, 1, 0, 'C');
            $pdf->Cell(143, 7, $aciklama, 1, 0, 'L');
            $pdf->Cell(30, 7, $tutar_str, 1, 1, 'R');
        } else {
            $pdf->Cell(15, 7, '', 'LR', 0, 'C');
            $pdf->MultiCell(173, 7, $satir, 'LRB', 'L');
        }
    }
} else {
    foreach ($teklif_detaylari as $index => $detay) {
        $urun_adi = $detay['urun_adi'] ?? '';
        $miktar = $detay['miktar'] ?? 0;
        $birim_fiyat = $detay['birim_fiyat'] ?? 0;
        $kdv_orani = $detay['kdv_orani'] ?? 0;
        $tutar = $detay['toplam_tutar'] ?? 0;

        $ara_toplam += $tutar;

        // Satır numarası
        $pdf->Cell(15, 7, $index + 1, 1, 0, 'C');
        // Ürün adı
        $pdf->Cell(75, 7, $urun_adi, 1, 0);
        // Miktar
        $pdf->Cell(22, 7, number_format($miktar, 2) . ' ' . ($detay['birim'] ?? ''), 1, 0, 'R');
        // Birim fiyat
        $pdf->Cell(28, 7, number_format($birim_fiyat, 2) . ' ' . ($detay['satis_para_birimi'] ?? $teklif['para_birimi'] ?? 'TRY'), 1, 0, 'R');
        // KDV
        $pdf->Cell(18, 7, number_format($kdv_orani, 0) . '%', 1, 0, 'C');
        // Tutar
        $pdf->Cell(30, 7, number_format($tutar, 2) . ' ' . ($teklif['para_birimi'] ?? 'TRY'), 1, 1, 'R');
    }
}

$pdf->Ln(10);

// Genel toplam hesaplama
$indirim_tutari = $teklif['indirim_tutari'] ?? 0;

// Ara toplam zaten ürün döngüsünde hesaplandı
// Teklif tablosundaki toplam tutarı kullan (eğer mevcutsa ve daha büyükse)
if (!empty($teklif['tutar']) && $teklif['tutar'] > $ara_toplam) {
    $ara_toplam = $teklif['tutar'];
}

// Genel toplam hesaplama
$genel_toplam = round($ara_toplam + $toplam_kdv - $indirim_tutari, 2);

// Toplamlar - Sağ tarafta düzenli
$pdf->SetFont('DejaVu', 'B', 10);
$pdf->SetFillColor(248, 249, 250);
$pdf->SetDrawColor(222, 226, 230);

// Toplamlar tablosu - sağ tarafta, daha geniş
$pdf->Cell(45, 10, '', 0, 0); // spacer to push totals right
$pdf->Cell(110, 10, __('subtotal') . ':', 1, 0, 'R', true);
$pdf->Cell(35, 10, number_format(round($ara_toplam, 2), 2) . ' ' . ($teklif['para_birimi'] ?? 'TRY'), 1, 1, 'R', true);

$pdf->Cell(45, 10, '', 0, 0); // spacer
$pdf->Cell(110, 10, __('total_vat') . ':', 1, 0, 'R', true);
$pdf->Cell(35, 10, number_format(round($toplam_kdv, 2), 2) . ' ' . ($teklif['para_birimi'] ?? 'TRY'), 1, 1, 'R', true);

if (!empty($teklif['indirim_tutari']) && $teklif['indirim_tutari'] > 0) {
    $pdf->Cell(45, 10, '', 0, 0); // spacer
    $pdf->Cell(110, 10, __('discount') . ':', 1, 0, 'R', true);
    $pdf->Cell(35, 10, '-' . number_format(round($indirim_tutari, 2), 2) . ' ' . ($teklif['para_birimi'] ?? 'TRY'), 1, 1, 'R', true);
}

// Genel toplam - mavi arka planla vurgulu
$pdf->SetFillColor(13, 110, 253);
$pdf->SetTextColor(255, 255, 255);
$pdf->SetFont('DejaVu', 'B', 12);
$pdf->Cell(45, 12, '', 0, 0); // spacer
$pdf->Cell(110, 12, strtoupper(__('grand_total')) . ':', 1, 0, 'R', true);
$pdf->Cell(35, 12, number_format($genel_toplam, 2) . ' ' . ($teklif['para_birimi'] ?? 'TRY'), 1, 1, 'R', true);

$pdf->Ln(15);

// Açıklama (sadece stoklu işlemlerde gösterilir)
if (!$is_stoksuz && !empty($teklif['aciklama'])) {
    $pdf->SetFont('DejaVu', 'B', 11);
    $pdf->SetTextColor(33, 37, 41);
    $pdf->Cell(0, 8, __('notes') . ':', 0, 1);
    
    $pdf->SetFont('DejaVu', '', 10);
    $pdf->SetTextColor(100, 100, 100);
    $pdf->MultiCell(0, 6, $teklif['aciklama'], 0, 'L');
} else {
    $pdf->Ln(10); // Not yoksa boşluk bırak
}

// Bilgi satırı hesaplama
$bilgi_satiri = str_replace(
    [':name', ':amount', ':days'],
    [
        $cari['ad_soyad'] ?? '',
        number_format($genel_toplam, 2, ',', '.') . ' ' . ($teklif['para_birimi'] ?? 'TRY'),
        $gecerlilik_gun
    ],
    $gecerlilik_gun > 0 ? __('quote_info_with_days') : __('quote_info')
);

$pdf->SetFont('DejaVu', '', 10);
$pdf->SetTextColor(100, 100, 100);
$pdf->Cell(0, 8, $bilgi_satiri, 0, 1, 'R');

// PDF'i göster
$pdf->Output('teklif_' . $teklif_id . '.pdf', 'I');
?>
?>