<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi('yonetici'); // Admin ve Yöneticiler erişebilir

$sayfa_baslik = "Teklif Düzenle";
$aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;

// Teklif ID'sini al ve doğrula
$teklif_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$teklif_id) {
    $_SESSION['hata'] = "Geçersiz teklif ID'si!";
    header("Location: teklifler.php");
    exit();
}

// Teklifi veritabanından al
try {
    $stmt_teklif = $db->prepare("SELECT t.*, c.ad_soyad as cari_adi FROM teklifler t 
                                 LEFT JOIN cariler c ON t.cari_id = c.id 
                                 WHERE t.id = ? AND t.sirket_id = ?");
    $stmt_teklif->execute([$teklif_id, $aktif_sirket_id]);
    $teklif = $stmt_teklif->fetch(PDO::FETCH_ASSOC);
    
    if (!$teklif) {
        $_SESSION['hata'] = "Teklif bulunamadı!";
        header("Location: teklifler.php");
        exit();
    }
    
    // Teklif tipini belirle (stoklu/stoksuz)
    $teklif_tipi = 'stoklu'; // Varsayılan olarak stoklu
    if (isset($teklif['teklif_tipi'])) {
        $teklif_tipi = $teklif['teklif_tipi'];
    } else {
        // Eğer teklif_tipi kolonu yoksa, teklif_detaylari tablosundan kontrol et
        $stmt_detay_kontrol = $db->prepare("SELECT COUNT(*) as detay_sayisi FROM teklif_detaylari WHERE teklif_id = ?");
        $stmt_detay_kontrol->execute([$teklif_id]);
        $detay_sayisi = $stmt_detay_kontrol->fetchColumn();
        
        if ($detay_sayisi == 0) {
            $teklif_tipi = 'stoksuz';
        }
    }
    
} catch (PDOException $e) {
    $_SESSION['hata'] = "Teklif bilgileri alınırken hata oluştu!";
    header("Location: teklifler.php");
    exit();
}

// Çoklu para birimi durumunu kontrol et
try {
    $stmt_coklu_pb = $db->prepare("SELECT coklu_para_birimi, para_birimi FROM sirketler WHERE id = ?");
    $stmt_coklu_pb->execute([$aktif_sirket_id]);
    $sirket_ayarlari = $stmt_coklu_pb->fetch(PDO::FETCH_ASSOC);
    $coklu_para_birimi_aktif = (bool) ($sirket_ayarlari['coklu_para_birimi'] ?? 0);
    $sirket_para_birimi = $sirket_ayarlari['para_birimi'] ?? 'TL';
} catch (PDOException $e) {
    $coklu_para_birimi_aktif = 0;
    $sirket_para_birimi = 'TL';
}

$paraBirimi = getParaBirimiSembolu(); // Para birimini al

// Veritabanından döviz kurlarını çek
$doviz_kurlari = [];
if ($coklu_para_birimi_aktif) {
    try {
        $stmt_doviz = $db->prepare("SELECT para_birimi, alis, satis, guncelleme_tarihi FROM doviz_kurlari ORDER BY id");
        $stmt_doviz->execute();
        $doviz_kurlari = $stmt_doviz->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $doviz_kurlari = [];
    }
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
    } else {
        $cari_id = filter_input(INPUT_POST, 'cari_id', FILTER_VALIDATE_INT);
        $teklif_tarihi = filter_input(INPUT_POST, 'teklif_tarihi', FILTER_SANITIZE_SPECIAL_CHARS);
        $gecerlilik_tarihi = filter_input(INPUT_POST, 'gecerlilik_tarihi', FILTER_SANITIZE_SPECIAL_CHARS);
        $aciklama = temizle($_POST['aciklama'] ?? '');
        $indirim_tutari = filter_var(str_replace(',', '.', $_POST['indirim_tutari'] ?? '0'), FILTER_VALIDATE_FLOAT);
        if ($indirim_tutari === false) $indirim_tutari = 0;
        
        // Teklif para birimi
        $teklif_para_birimi = temizle($_POST['teklif_para_birimi'] ?? 'TL');
        if ($teklif_para_birimi === 'TL') $teklif_para_birimi = 'TRY';

        $urun_detaylari = [];
        $toplam_tutar = 0;
        $toplam_kdv = 0;

        if ($teklif_tipi == 'stoklu') {
            // Stoklu teklifler için ürün ID'leri ile işle
            $urun_ids = $_POST['urun_id'] ?? [];
            $miktarlar = $_POST['miktar'] ?? [];
            $birim_fiyatlar = $_POST['birim_fiyat'] ?? [];
            $kdv_oranlari = $_POST['kdv_orani'] ?? [];
            $kurlar = $_POST['kur'] ?? [];
            $satis_para_birimleri = $_POST['satis_para_birimi'] ?? [];

            foreach ($urun_ids as $index => $urun_id) {
                if (!empty($urun_id) && !empty($miktarlar[$index]) && isset($birim_fiyatlar[$index])) {
                    $miktar = filter_var(str_replace(',', '.', $miktarlar[$index]), FILTER_VALIDATE_FLOAT);
                    $birim_fiyat = filter_var(str_replace(',', '.', $birim_fiyatlar[$index]), FILTER_VALIDATE_FLOAT);
                    $kdv_orani = filter_var(str_replace(',', '.', $kdv_oranlari[$index]), FILTER_VALIDATE_FLOAT);
                    $kur = filter_var(str_replace(',', '.', $kurlar[$index] ?? '1'), FILTER_VALIDATE_FLOAT);
                    $satis_para_birimi = temizle($satis_para_birimleri[$index] ?? 'TL');

                    if ($miktar > 0 && $birim_fiyat >= 0) {
                        $satir_toplam_orj = $miktar * $birim_fiyat;
                        $kdv_tutari_orj = ($satir_toplam_orj * $kdv_orani) / 100;

                        $satir_toplam_teklif_pb = $satir_toplam_orj * $kur;
                        $kdv_tutari_teklif_pb = $kdv_tutari_orj * $kur;

                        $urun_detaylari[] = [
                            'urun_id' => $urun_id, 
                            'miktar' => $miktar, 
                            'birim_fiyat' => $birim_fiyat,
                            'toplam_tutar' => $satir_toplam_teklif_pb,
                            'kdv_orani' => $kdv_orani, 
                            'kdv_tutari' => $kdv_tutari_teklif_pb,
                            'kur' => $kur,
                            'satis_para_birimi' => $satis_para_birimi
                        ];
                        
                        $toplam_tutar += $satir_toplam_teklif_pb;
                        $toplam_kdv += $kdv_tutari_teklif_pb;
                    }
                }
            }
        } else {
            // Stoksuz teklifler için kalem açıklamaları ile işle
            $kalem_aciklamalari = $_POST['kalem_aciklama'] ?? [];
            $miktarlar = $_POST['miktar'] ?? [];
            $birim_fiyatlar = $_POST['birim_fiyat'] ?? [];
            $kdv_oranlari = $_POST['kdv_orani'] ?? [];

            foreach ($kalem_aciklamalari as $index => $kalem_aciklama) {
                if (!empty($kalem_aciklama) && !empty($miktarlar[$index]) && isset($birim_fiyatlar[$index])) {
                    $miktar = filter_var(str_replace(',', '.', $miktarlar[$index]), FILTER_VALIDATE_FLOAT);
                    $birim_fiyat = filter_var(str_replace(',', '.', $birim_fiyatlar[$index]), FILTER_VALIDATE_FLOAT);
                    $kdv_orani = filter_var(str_replace(',', '.', $kdv_oranlari[$index]), FILTER_VALIDATE_FLOAT);

                    if ($miktar > 0 && $birim_fiyat >= 0) {
                        $satir_toplam = $miktar * $birim_fiyat;
                        $kdv_tutari = ($satir_toplam * $kdv_orani) / 100;

                        $urun_detaylari[] = [
                            'kalem_aciklama' => $kalem_aciklama,
                            'miktar' => $miktar, 
                            'birim_fiyat' => $birim_fiyat,
                            'toplam_tutar' => $satir_toplam,
                            'kdv_orani' => $kdv_orani, 
                            'kdv_tutari' => $kdv_tutari
                        ];
                        
                        $toplam_tutar += $satir_toplam;
                        $toplam_kdv += $kdv_tutari;
                    }
                }
            }
        }
        
        $hatalar = [];
        if (empty($cari_id)) $hatalar[] = "Müşteri seçimi zorunludur.";
        if (empty($teklif_tarihi)) $hatalar[] = "Teklif tarihi zorunludur.";
        if (empty($urun_detaylari)) $hatalar[] = "Teklif için en az bir ürün eklemelisiniz.";

        if (empty($hatalar)) {
            try {
                $db->beginTransaction();
                
                if ($teklif_tipi == 'stoklu') {
                    // Stoklu teklif güncelleme
                $stmt_guncelle = $db->prepare("UPDATE teklifler SET 
                    cari_id = ?, teklif_tarihi = ?, gecerlilik_tarihi = ?, 
                        tutar = ?, toplam_kdv = ?, indirim_tutari = ?, aciklama = ?, teklif_tipi = 'stoklu'
                    WHERE id = ? AND sirket_id = ?");
                $stmt_guncelle->execute([
                    $cari_id, $teklif_tarihi, empty($gecerlilik_tarihi) ? null : $gecerlilik_tarihi,
                    $toplam_tutar, $toplam_kdv, $indirim_tutari, $aciklama,
                    $teklif_id, $aktif_sirket_id
                ]);

                // Mevcut teklif kalemlerini sil
                    $stmt_sil = $db->prepare("DELETE FROM teklif_detaylari WHERE teklif_id = ? AND sirket_id = ?");
                    $stmt_sil->execute([$teklif_id, $aktif_sirket_id]);

                // Yeni teklif kalemlerini ekle
                $stmt_ekle = $db->prepare("INSERT INTO teklif_detaylari 
                    (teklif_id, urun_id, miktar, birim_fiyat, kdv_orani, kdv_tutari, toplam_tutar, sirket_id, para_birimi, kur) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                
                foreach ($urun_detaylari as $kalem) {
                    $stmt_ekle->execute([
                        $teklif_id, $kalem['urun_id'], $kalem['miktar'], 
                        $kalem['birim_fiyat'], $kalem['kdv_orani'], 
                        $kalem['kdv_tutari'], $kalem['toplam_tutar'], 
                        $aktif_sirket_id, $kalem['satis_para_birimi'], $kalem['kur']
                    ]);
            }
                        } else {
                    // Stoksuz teklif güncelleme - kalemleri açıklama alanına ekle
                    $stoksuz_aciklama = $aciklama . "\n\n";
                    foreach ($urun_detaylari as $kalem) {
                        $stoksuz_aciklama .= $kalem['kalem_aciklama'] . " - " . 
                                            $kalem['miktar'] . " adet x " . 
                                            number_format($kalem['birim_fiyat'], 2, ',', '.') . " TL = " . 
                                            number_format($kalem['toplam_tutar'], 2, ',', '.') . " TL\n";
                    }
                    
                    $stmt_guncelle = $db->prepare("UPDATE teklifler SET 
                        cari_id = ?, teklif_tarihi = ?, gecerlilik_tarihi = ?, 
                        tutar = ?, toplam_kdv = ?, indirim_tutari = ?, aciklama = ?, teklif_tipi = 'stoksuz'
                        WHERE id = ? AND sirket_id = ?");
                    $stmt_guncelle->execute([
                        $cari_id, $teklif_tarihi, empty($gecerlilik_tarihi) ? null : $gecerlilik_tarihi,
                        $toplam_tutar, $toplam_kdv, $indirim_tutari, $stoksuz_aciklama,
                        $teklif_id, $aktif_sirket_id
                    ]);
                }

                $db->commit();

                // LOGLAMA
                $stmt_cari_ad = $db->prepare("SELECT ad_soyad FROM cariler WHERE id = ? AND sirket_id = ?");
                $stmt_cari_ad->execute([$cari_id, $aktif_sirket_id]);
                $cari_adi = $stmt_cari_ad->fetchColumn();
                $genel_toplam = $toplam_tutar - $indirim_tutari;
                $log_mesaji = "kullanıcısı, #{$teklif_id} ID'li teklifi güncelledi. Müşteri: {$cari_adi}, Tutar: " . number_format($genel_toplam, 2) . " $paraBirimi";
                sistem_log_ekle($db, 'guncelle', 'teklifler', $teklif_id, $teklif, ['yeni_veriler' => $_POST], $log_mesaji);

                $_SESSION['mesaj'] = "Teklif #{$teklif_id} başarıyla güncellendi.";
                header("Location: teklifler.php");
                exit();
            } catch (Exception $e) {
                $db->rollBack();
                $_SESSION['hata'] = "Teklif güncellenirken bir hata oluştu: " . $e->getMessage();
            }
        } else {
            $_SESSION['form_hatalari'] = $hatalar;
            $_SESSION['form_verileri'] = $_POST;
        }
    }
    header("Location: teklif_duzenle.php?id=" . $teklif_id);
    exit();
}

// Cariler sorgusu
$stmt_cariler = $db->prepare("SELECT id, ad_soyad FROM cariler WHERE sirket_id = ? ORDER BY ad_soyad");
$stmt_cariler->execute([$aktif_sirket_id]);
$cariler = $stmt_cariler->fetchAll(PDO::FETCH_ASSOC);

// Mevcut form verilerini al
$form_verileri = $_SESSION['form_verileri'] ?? [];
$form_hatalari = $_SESSION['form_hatalari'] ?? [];
unset($_SESSION['form_verileri'], $_SESSION['form_hatalari']);

// Mevcut teklif kalemlerini al
$teklif_kalemleri = [];

if ($teklif_tipi == 'stoklu') {
    // Stoklu teklifler için teklif_detaylari tablosundan kalemleri al
    try {
        $stmt_kalemler = $db->prepare("SELECT td.*, u.urun_adi, u.stok_kodu, u.stok_miktari, u.birim_fiyat as urun_birim_fiyat, u.kdv_orani as urun_kdv_orani, u.para_birimi as urun_para_birimi
                                       FROM teklif_detaylari td 
                                       LEFT JOIN urunler u ON td.urun_id = u.id
                                       WHERE td.teklif_id = ? AND td.sirket_id = ? 
                                       ORDER BY td.id");
        $stmt_kalemler->execute([$teklif_id, $aktif_sirket_id]);
    $teklif_kalemleri = $stmt_kalemler->fetchAll(PDO::FETCH_ASSOC);
    
        error_log("Stoklu teklif ID: $teklif_id için " . count($teklif_kalemleri) . " kalem bulundu");
        
    } catch (PDOException $e) {
        error_log("Stoklu teklif kalemleri alınırken hata: " . $e->getMessage());
        $teklif_kalemleri = [];
    }
} else {
    // Stoksuz teklifler için teklif açıklamasından parse et
    if (!empty($teklif['aciklama'])) {
        error_log("Stoksuz teklif açıklamasından kalemler parse ediliyor: " . substr($teklif['aciklama'], 0, 100) . "...");
        
        // Teklif açıklamasını satır satır parse et
        $satirlar = explode("\n", $teklif['aciklama']);
        foreach ($satirlar as $index => $satir) {
            $satir = trim($satir);
            if (empty($satir)) continue;
            
            // Ürün adı ve fiyat bilgisi varsa kalem olarak ekle
            if (strpos($satir, 'adet') !== false || strpos($satir, 'TL') !== false || strpos($satir, '$') !== false || strpos($satir, '€') !== false || strpos($satir, 'x') !== false) {
                // Fiyat bilgisini çıkar
                preg_match('/(\d+[.,]\d+|\d+)\s*(TL|\$|€)/', $satir, $fiyat_eslesme);
                $fiyat = isset($fiyat_eslesme[1]) ? str_replace(',', '.', $fiyat_eslesme[1]) : 0;
                
                // Miktar bilgisini çıkar
                preg_match('/(\d+)\s*(adet|x|pcs)/', $satir, $miktar_eslesme);
                $miktar = isset($miktar_eslesme[1]) ? $miktar_eslesme[1] : 1;
                
                $teklif_kalemleri[] = [
                    'urun_id' => null, // Stoksuz tekliflerde ürün ID yok
                    'urun_adi' => $satir,
                    'stok_kodu' => 'KALEM-' . ($index + 1),
                    'miktar' => $miktar,
                    'birim_fiyat' => $fiyat > 0 ? $fiyat / $miktar : 0,
                    'kdv_orani' => 0,
                    'kur' => 1,
                    'satis_para_birimi' => 'TL'
                ];
            }
        }
        
        error_log("Stoksuz teklif parse edilen kalem sayısı: " . count($teklif_kalemleri));
    }
}

// Ürün arama için AJAX endpoint'i
$urunler = [];
if (!empty($form_verileri['urun_arama'] ?? '')) {
    try {
        $stmt_urunler = $db->prepare("SELECT id, urun_adi, stok_kodu, stok_miktari, birim_fiyat, kdv_orani, para_birimi 
                                     FROM urunler WHERE sirket_id = ? AND (urun_adi LIKE ? OR stok_kodu LIKE ?) 
                                     ORDER BY urun_adi LIMIT 20");
        $arama_terimi = '%' . ($form_verileri['urun_arama'] ?? '') . '%';
        $stmt_urunler->execute([$aktif_sirket_id, $arama_terimi, $arama_terimi]);
        $urunler = $stmt_urunler->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $urunler = [];
    }
}

// CSRF token oluştur
$csrf_token = csrf_token_olustur();
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $sayfa_baslik ?> - Odjoo</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="bi bi-pencil-square"></i> 
                            Teklif Düzenle #<?= $teklif_id ?>
                        </h5>
                        <a href="teklifler.php" class="btn btn-secondary">
                            <i class="bi bi-arrow-left"></i> Geri Dön
                        </a>
                    </div>
                    <div class="card-body">
                                                 <?php if (!empty($form_hatalari)): ?>
                             <div class="alert alert-danger">
                                 <ul class="mb-0">
                                     <?php foreach ($form_hatalari as $hata): ?>
                                         <li><?= htmlspecialchars($hata) ?></li>
                                     <?php endforeach; ?>
                                 </ul>
                             </div>
                         <?php endif; ?>

                                                   <!-- Debug bilgisi -->
                          <div class="alert alert-info">
                              <strong>Debug:</strong> Teklif ID: <?= $teklif_id ?>, 
                              Teklif Tipi: <?= $teklif_tipi ?>,
                              Kalem Sayısı: <?= count($teklif_kalemleri) ?>,
                              Teklif Tutar: <?= $teklif['tutar'] ?? 'N/A' ?>,
                              Teklif KDV: <?= $teklif['toplam_kdv'] ?? 'N/A' ?>,
                              Aktif Şirket ID: <?= $aktif_sirket_id ?>,
                              Teklif Şirket ID: <?= $teklif['sirket_id'] ?? 'N/A' ?>
                          </div>

                        <form method="POST" id="teklifForm">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="cari_id" class="form-label">Müşteri *</label>
                                        <select name="cari_id" id="cari_id" class="form-select" required>
                                            <option value="">Müşteri Seçin</option>
                                            <?php foreach ($cariler as $cari): ?>
                                                <option value="<?= $cari['id'] ?>" 
                                                    <?= ($form_verileri['cari_id'] ?? $teklif['cari_id']) == $cari['id'] ? 'selected' : '' ?>>
                                                    <?= htmlspecialchars($cari['ad_soyad']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="teklif_tarihi" class="form-label">Teklif Tarihi *</label>
                                        <input type="date" class="form-control" id="teklif_tarihi" name="teklif_tarihi" 
                                               value="<?= htmlspecialchars($form_verileri['teklif_tarihi'] ?? $teklif['teklif_tarihi']) ?>" required>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="gecerlilik_tarihi" class="form-label">Geçerlilik Tarihi</label>
                                        <input type="date" class="form-control" id="gecerlilik_tarihi" name="gecerlilik_tarihi" 
                                               value="<?= htmlspecialchars($form_verileri['gecerlilik_tarihi'] ?? $teklif['gecerlilik_tarihi']) ?>">
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="teklif_para_birimi" class="form-label">Teklif Para Birimi</label>
                                        <select name="teklif_para_birimi" id="teklif_para_birimi" class="form-select">
                                            <option value="TL" <?= ($form_verileri['teklif_para_birimi'] ?? 'TL') == 'TL' ? 'selected' : '' ?>>TL</option>
                                            <?php if ($coklu_para_birimi_aktif): ?>
                                                <option value="USD" <?= ($form_verileri['teklif_para_birimi'] ?? 'TL') == 'USD' ? 'selected' : '' ?>>USD</option>
                                                <option value="EUR" <?= ($form_verileri['teklif_para_birimi'] ?? 'TL') == 'EUR' ? 'selected' : '' ?>>EUR</option>
                                            <?php endif; ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="indirim_tutari" class="form-label">İndirim Tutarı</label>
                                        <input type="text" class="form-control" id="indirim_tutari" name="indirim_tutari" 
                                               value="<?= htmlspecialchars($form_verileri['indirim_tutari'] ?? number_format($teklif['indirim_tutari'] ?? 0, 2, ',', '.')) ?>">
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="aciklama" class="form-label">Genel Açıklama</label>
                                <textarea class="form-control" id="aciklama" name="aciklama" rows="3"><?= htmlspecialchars($form_verileri['aciklama'] ?? $teklif['aciklama']) ?></textarea>
                            </div>

                                                         <hr>
                             <h6>Ürün Kalemleri 
                                 <span class="badge bg-<?= $teklif_tipi == 'stoklu' ? 'success' : 'warning' ?>">
                                     <?= $teklif_tipi == 'stoklu' ? 'Stoklu Teklif' : 'Stoksuz Teklif' ?>
                                 </span>
                             </h6>
                             
                             <?php if ($teklif_tipi == 'stoklu'): ?>
                             <div class="mb-3">
                                 <button type="button" class="btn btn-primary" id="addUrunKalem">
                                     <i class="bi bi-plus-circle"></i> Ürün Ekle
                                 </button>
                                 <button type="button" class="btn btn-info" data-bs-toggle="modal" data-bs-target="#urunEkleModal">
                                     <i class="bi bi-search"></i> Ürün Ara
                                 </button>
                             </div>
                             <?php else: ?>
                             <div class="mb-3">
                                 <button type="button" class="btn btn-primary" id="addStoksuzKalem">
                                     <i class="bi bi-plus-circle"></i> Kalem Ekle
                                 </button>
                             </div>
                             <?php endif; ?>
                             
                             <div class="table-responsive">
                                 <table class="table table-bordered table-hover">
                                     <thead class="table-light">
                                         <tr>
                                             <?php if ($teklif_tipi == 'stoklu'): ?>
                                             <th>Ürün/Hizmet</th>
                                             <th width="100">Adet</th>
                                             <th width="120">Birim Fiyat</th>
                                             <th width="100">Ürün PB</th>
                                             <th width="100">Kur</th>
                                             <th width="80">KDV (%)</th>
                                             <th width="120">KDV Tutarı</th>
                                             <th width="120">Toplam</th>
                                             <th width="60">Sil</th>
                                             <?php else: ?>
                                             <th>Kalem Açıklaması</th>
                                             <th width="100">Adet</th>
                                             <th width="120">Birim Fiyat</th>
                                             <th width="80">KDV (%)</th>
                                             <th width="120">KDV Tutarı</th>
                                             <th width="120">Toplam</th>
                                             <th width="60">Sil</th>
                                             <?php endif; ?>
                                         </tr>
                                     </thead>
                                     <tbody id="urunKalemleri">
                                         <?php if (!empty($teklif_kalemleri)): ?>
                                             <?php foreach ($teklif_kalemleri as $index => $kalem): ?>
                                                 <tr class="urun-kalem">
                                                     <?php if ($teklif_tipi == 'stoklu'): ?>
                                                     <td>
                                                         <select name="urun_id[]" class="form-select urun-select" required>
                                                             <option value="">Ürün Seçin</option>
                                                             <?php if ($kalem['urun_id']): ?>
                                                             <option value="<?= $kalem['urun_id'] ?>" selected>
                                                                 <?= htmlspecialchars($kalem['urun_adi'] ?? 'Ürün ID: ' . $kalem['urun_id']) ?>
                                                                 <?php if ($kalem['stok_kodu']): ?>
                                                                 (<?= htmlspecialchars($kalem['stok_kodu']) ?>)
                                                                 <?php endif; ?>
                                                             </option>
                                                             <?php endif; ?>
                                                         </select>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="miktar[]" class="form-control miktar-input text-center" 
                                                                value="<?= htmlspecialchars($kalem['miktar']) ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="birim_fiyat[]" class="form-control birim-fiyat-input text-end" 
                                                                value="<?= htmlspecialchars(number_format($kalem['birim_fiyat'], 2, ',', '.')) ?>" required>
                                                     </td>
                                                     <td>
                                                         <select name="satis_para_birimi[]" class="form-select para-birimi-select">
                                                             <option value="TL" <?= ($kalem['satis_para_birimi'] ?? 'TL') == 'TL' ? 'selected' : '' ?>>TL</option>
                                                             <?php if ($coklu_para_birimi_aktif): ?>
                                                                 <option value="USD" <?= ($kalem['satis_para_birimi'] ?? 'TL') == 'USD' ? 'selected' : '' ?>>USD</option>
                                                                 <option value="EUR" <?= ($kalem['satis_para_birimi'] ?? 'TL') == 'EUR' ? 'selected' : '' ?>>EUR</option>
                                                             <?php endif; ?>
                                                         </select>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="kur[]" class="form-control kur-input text-center" 
                                                                value="<?= htmlspecialchars($kalem['kur'] ?? '1') ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="kdv_orani[]" class="form-control kdv-input text-center" 
                                                                value="<?= htmlspecialchars($kalem['kdv_orani'] ?? '0') ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" class="form-control kdv-tutari-display text-end" 
                                                                value="<?= htmlspecialchars(number_format(($kalem['miktar'] * $kalem['birim_fiyat'] * ($kalem['kur'] ?? 1) * ($kalem['kdv_orani'] ?? 0)) / 100, 2, ',', '.')) ?>" readonly>
                                                     </td>
                                                     <td>
                                                         <input type="text" class="form-control satir-toplam-display text-end" 
                                                                value="<?= htmlspecialchars(number_format($kalem['miktar'] * $kalem['birim_fiyat'] * ($kalem['kur'] ?? 1), 2, ',', '.')) ?>" readonly>
                                                     </td>
                                         <?php else: ?>
                                                     <td>
                                                         <input type="text" name="kalem_aciklama[]" class="form-control" 
                                                                value="<?= htmlspecialchars($kalem['urun_adi'] ?? '') ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="miktar[]" class="form-control miktar-input text-center" 
                                                                value="<?= htmlspecialchars($kalem['miktar']) ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="birim_fiyat[]" class="form-control birim-fiyat-input text-end" 
                                                                value="<?= htmlspecialchars(number_format($kalem['birim_fiyat'], 2, ',', '.')) ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" name="kdv_orani[]" class="form-control kdv-input text-center" 
                                                                value="<?= htmlspecialchars($kalem['kdv_orani'] ?? '0') ?>" required>
                                                     </td>
                                                     <td>
                                                         <input type="text" class="form-control kdv-tutari-display text-end" 
                                                                value="<?= htmlspecialchars(number_format(($kalem['miktar'] * $kalem['birim_fiyat'] * ($kalem['kdv_orani'] ?? 0)) / 100, 2, ',', '.')) ?>" readonly>
                                                     </td>
                                                     <td>
                                                         <input type="text" class="form-control satir-toplam-display text-end" 
                                                                value="<?= htmlspecialchars(number_format($kalem['miktar'] * $kalem['birim_fiyat'], 2, ',', '.')) ?>" readonly>
                                                     </td>
                                                     <?php endif; ?>
                                                     <td class="text-center">
                                                         <button type="button" class="btn btn-danger btn-sm remove-kalem" title="Kalemi Kaldır">
                                                             <i class="bi bi-trash"></i>
                                                         </button>
                                                     </td>
                                                 </tr>
                                             <?php endforeach; ?>
                                         <?php else: ?>
                                             <tr class="urun-kalem">
                                                 <?php if ($teklif_tipi == 'stoklu'): ?>
                                                 <td>
                                                     <select name="urun_id[]" class="form-select urun-select" required>
                                                         <option value="">Ürün Seçin</option>
                                                     </select>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="miktar[]" class="form-control miktar-input text-center" value="1" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="birim_fiyat[]" class="form-control birim-fiyat-input text-end" required>
                                                 </td>
                                                 <td>
                                                     <select name="satis_para_birimi[]" class="form-select para-birimi-select">
                                                         <option value="TL" selected>TL</option>
                                                         <?php if ($coklu_para_birimi_aktif): ?>
                                                             <option value="USD">USD</option>
                                                             <option value="EUR">EUR</option>
                                                         <?php endif; ?>
                                                     </select>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="kur[]" class="form-control kur-input text-center" value="1" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="kdv_orani[]" class="form-control kdv-input text-center" value="0" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" class="form-control kdv-tutari-display text-end" value="0,00" readonly>
                                                 </td>
                                                 <td>
                                                     <input type="text" class="form-control satir-toplam-display text-end" value="0,00" readonly>
                                                 </td>
                                                 <?php else: ?>
                                                 <td>
                                                     <input type="text" name="kalem_aciklama[]" class="form-control" placeholder="Kalem açıklaması girin" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="miktar[]" class="form-control miktar-input text-center" value="1" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="birim_fiyat[]" class="form-control birim-fiyat-input text-end" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" name="kdv_orani[]" class="form-control kdv-input text-center" value="0" required>
                                                 </td>
                                                 <td>
                                                     <input type="text" class="form-control kdv-tutari-display text-end" value="0,00" readonly>
                                                 </td>
                                                 <td>
                                                     <input type="text" class="form-control satir-toplam-display text-end" value="0,00" readonly>
                                                 </td>
                                                 <?php endif; ?>
                                                 <td class="text-center">
                                                     <button type="button" class="btn btn-danger btn-sm remove-kalem" title="Kalemi Kaldır">
                                                         <i class="bi bi-trash"></i>
                                                     </button>
                                                 </td>
                                             </tr>
                                         <?php endif; ?>
                                     </tbody>
                                 </table>
                             </div>

                            

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="alert alert-info">
                                        <strong>Ara Toplam:</strong> <span id="araToplam">0,00</span> <?= $paraBirimi ?>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="alert alert-warning">
                                        <strong>Toplam KDV:</strong> <span id="toplamKdv">0,00</span> <?= $paraBirimi ?>
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="alert alert-success">
                                        <strong>Genel Toplam:</strong> <span id="genelToplam">0,00</span> <?= $paraBirimi ?>
                                    </div>
                                </div>
                            </div>

                            <div class="d-flex justify-content-end gap-2">
                                <a href="teklifler.php" class="btn btn-secondary">
                                    <i class="bi bi-x-circle"></i> İptal
                                </a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-check-circle"></i> Teklifi Güncelle
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Ürün Ekleme Modal -->
    <div class="modal fade" id="urunEkleModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Ürün Ekle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <input type="text" class="form-control" id="urunArama" placeholder="Ürün adı veya stok kodu ile arama yapın...">
                    </div>
                    <div id="urunSonuclar"></div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const urunKalemleri = document.getElementById('urunKalemleri');
            const addUrunKalem = document.getElementById('addUrunKalem');
            const urunArama = document.getElementById('urunArama');
            const urunSonuclar = document.getElementById('urunSonuclar');

                         // Yeni ürün kalemi ekle (stoklu teklifler için)
             addUrunKalem.addEventListener('click', function() {
                 const yeniKalem = document.querySelector('.urun-kalem').cloneNode(true);
                 yeniKalem.querySelectorAll('input, select').forEach(input => {
                     if (input.classList.contains('kdv-tutari-display') || input.classList.contains('satir-toplam-display')) {
                         input.value = '0,00';
                     } else if (input.classList.contains('kur-input')) {
                         input.value = '1';
                     } else if (input.classList.contains('para-birimi-select')) {
                         input.value = 'TL';
                     } else if (input.classList.contains('miktar-input')) {
                         input.value = '1';
                     } else if (input.classList.contains('kdv-input')) {
                         input.value = '0';
                     } else {
                         input.value = '';
                     }
                 });
                 urunKalemleri.appendChild(yeniKalem);
                 addRowListeners(yeniKalem);
                 hesaplaToplamlar();
             });

             // Stoksuz teklifler için kalem ekleme
             const addStoksuzKalem = document.getElementById('addStoksuzKalem');
             if (addStoksuzKalem) {
                 addStoksuzKalem.addEventListener('click', function() {
                     const yeniKalem = document.querySelector('.urun-kalem').cloneNode(true);
                     yeniKalem.querySelectorAll('input, select').forEach(input => {
                         if (input.classList.contains('kdv-tutari-display') || input.classList.contains('satir-toplam-display')) {
                             input.value = '0,00';
                         } else if (input.classList.contains('miktar-input')) {
                             input.value = '1';
                         } else if (input.classList.contains('kdv-input')) {
                             input.value = '0';
                         } else {
                             input.value = '';
                         }
                     });
                     urunKalemleri.appendChild(yeniKalem);
                     addRowListeners(yeniKalem);
                     hesaplaToplamlar();
                 });
             }

            // Kalem kaldır
            document.addEventListener('click', function(e) {
                if (e.target.closest('.remove-kalem')) {
                    if (document.querySelectorAll('.urun-kalem').length > 1) {
                        e.target.closest('.urun-kalem').remove();
                        hesaplaToplamlar();
                    } else {
                        alert('En az bir ürün kalemi bulunmalıdır.');
                    }
                }
            });

            // Satır dinleyicileri ekle
            function addRowListeners(row) {
                row.querySelectorAll('input').forEach(input => {
                    input.addEventListener('input', hesaplaToplamlar);
                });
                row.querySelectorAll('select').forEach(select => {
                    select.addEventListener('change', hesaplaToplamlar);
                });
            }

            // Mevcut satırlara dinleyici ekle
            document.querySelectorAll('.urun-kalem').forEach(row => {
                addRowListeners(row);
            });

                         // Toplamları hesapla
             function hesaplaToplamlar() {
                 let araToplam = 0;
                 let toplamKdv = 0;
 
                 document.querySelectorAll('.urun-kalem').forEach(kalem => {
                     const miktar = parseFloat(kalem.querySelector('.miktar-input').value.replace(/\./g, '').replace(',', '.')) || 0;
                     const birimFiyat = parseFloat(kalem.querySelector('.birim-fiyat-input').value.replace(/\./g, '').replace(',', '.')) || 0;
                     const kdvOrani = parseFloat(kalem.querySelector('.kdv-input').value.replace(/\./g, '').replace(',', '.')) || 0;
                     
                     // Stoklu tekliflerde kur kullan, stoksuz tekliflerde kur = 1
                     let kur = 1;
                     const kurInput = kalem.querySelector('.kur-input');
                     if (kurInput) {
                         kur = parseFloat(kurInput.value.replace(/\./g, '').replace(',', '.')) || 1;
                     }
 
                     const satirToplam = miktar * birimFiyat * kur;
                     const kdvTutari = (satirToplam * kdvOrani) / 100;
 
                     // Satır toplamlarını güncelle
                     kalem.querySelector('.satir-toplam-display').value = formatCurrency(satirToplam);
                     kalem.querySelector('.kdv-tutari-display').value = formatCurrency(kdvTutari);
 
                     araToplam += satirToplam;
                     toplamKdv += kdvTutari;
                 });
 
                 const indirimTutari = parseFloat(document.getElementById('indirim_tutari').value.replace(/\./g, '').replace(',', '.')) || 0;
                 const genelToplam = araToplam + toplamKdv - indirimTutari;
 
                 document.getElementById('araToplam').textContent = formatCurrency(araToplam);
                 document.getElementById('toplamKdv').textContent = formatCurrency(toplamKdv);
                 document.getElementById('genelToplam').textContent = formatCurrency(genelToplam);
             }

            // Para formatı
            function formatCurrency(value) {
                return value.toLocaleString('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
            }

            // İndirim tutarı değiştiğinde toplamları güncelle
            document.getElementById('indirim_tutari').addEventListener('input', hesaplaToplamlar);

            // Sayfa yüklendiğinde toplamları hesapla
            hesaplaToplamlar();

            // Ürün arama
            urunArama.addEventListener('input', function() {
                const arama = this.value.trim();
                if (arama.length >= 2) {
                    fetch('../ajax/urun_ara_ajax.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: 'arama=' + encodeURIComponent(arama)
                    })
                    .then(response => response.json())
                    .then(data => {
                        let html = '';
                        if (data.success && data.urunler.length > 0) {
                            data.urunler.forEach(urun => {
                                html += `<div class="urun-item p-2 border-bottom" data-urun-id="${urun.id}" data-urun-adi="${urun.urun_adi}" data-stok-kodu="${urun.stok_kodu}" data-birim-fiyat="${urun.birim_fiyat}" data-kdv-orani="${urun.kdv_orani}" data-para-birimi="${urun.para_birimi}">
                                    <strong>${urun.urun_adi}</strong><br>
                                    <small class="text-muted">${urun.stok_kodu} - Stok: ${urun.stok_miktari}</small>
                                </div>`;
                            });
                        } else {
                            html = '<div class="text-muted p-2">Ürün bulunamadı.</div>';
                        }
                        urunSonuclar.innerHTML = html;
                    });
                } else {
                    urunSonuclar.innerHTML = '';
                }
            });

            // Ürün seç
            urunSonuclar.addEventListener('click', function(e) {
                if (e.target.closest('.urun-item')) {
                    const urunItem = e.target.closest('.urun-item');
                    const urunId = urunItem.dataset.urunId;
                    const urunAdi = urunItem.dataset.urunAdi;
                    const stokKodu = urunItem.dataset.stokKodu;
                    const birimFiyat = urunItem.dataset.birimFiyat;
                    const kdvOrani = urunItem.dataset.kdvOrani;
                    const paraBirimi = urunItem.dataset.paraBirimi;

                    // Boş kalem bul ve doldur
                    const bosKalem = document.querySelector('.urun-kalem:not(:has(.urun-select option:checked))');
                    if (bosKalem) {
                        bosKalem.querySelector('.urun-select').innerHTML = `<option value="${urunId}" selected>${urunAdi} (${stokKodu})</option>`;
                        bosKalem.querySelector('.birim-fiyat-input').value = formatCurrency(birimFiyat);
                        bosKalem.querySelector('.kdv-input').value = kdvOrani;
                        bosKalem.querySelector('.para-birimi-select').value = paraBirimi;
                        hesaplaToplamlar();
                    }

                    // Modal'ı kapat
                    const modal = bootstrap.Modal.getInstance(document.getElementById('urunEkleModal'));
                    modal.hide();
                }
            });
        });
    </script>

    <?php include __DIR__ . '/../includes/footer.php'; ?>
</body>
</html>
