<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/ayar_fonksiyonlari.php';
yetkili_mi('raporlar');

$sayfa_baslik = __("reports");
$paraBirimi = getParaBirimiSembolu(); // Para birimini al

// Tarih aralığı filtreleme için varsayılan değerler
$baslangic_tarihi = isset($_GET['baslangic_tarihi']) ? $_GET['baslangic_tarihi'] : date('Y-m-01'); // Ayın başı
$bitis_tarihi = isset($_GET['bitis_tarihi']) ? $_GET['bitis_tarihi'] : date('Y-m-d'); // Bugün

// Kategori bazında borç raporu
try {
    // Tarih aralığına göre filtrelenmiş kategori borçları
    $stmt_kategori = $db->prepare("
        SELECT k.kategori_adi, COALESCE(SUM(b.tutar), 0) as toplam_borc
        FROM kategoriler k
        LEFT JOIN borclar b ON k.id = b.kategori_id AND b.tarih BETWEEN :baslangic AND :bitis AND b.sirket_id = :sirket_id_join
        WHERE k.sirket_id = :sirket_id_where
        GROUP BY k.kategori_adi
        ORDER BY toplam_borc DESC
    ");
    $stmt_kategori->execute([
        ':baslangic' => $baslangic_tarihi, 
        ':bitis' => $bitis_tarihi, 
        ':sirket_id_join' => $aktif_sirket_id,
        ':sirket_id_where' => $aktif_sirket_id
    ]);
    $kategori_borclari = $stmt_kategori->fetchAll(PDO::FETCH_ASSOC);
    
    // Cari grup bazında bakiye raporu
    $stmt_grup = $db->prepare("
        SELECT g.grup_adi, COALESCE(SUM(c.bakiye), 0) as toplam_bakiye
        FROM cari_gruplari g
        LEFT JOIN cariler c ON g.id = c.grup_id AND c.sirket_id = :sirket_id_join
        WHERE g.sirket_id = :sirket_id_where
        GROUP BY g.grup_adi
        ORDER BY toplam_bakiye DESC
    ");
    $stmt_grup->execute([
        ':sirket_id_join' => $aktif_sirket_id,
        ':sirket_id_where' => $aktif_sirket_id
    ]);
    $grup_bakiyeleri = $stmt_grup->fetchAll(PDO::FETCH_ASSOC);
    
    // Aylık gelir-gider raporu
    $stmt_gelir_gider = $db->prepare("
        SELECT 
            SUM(CASE WHEN h.tur = 'giris' THEN h.tutar ELSE 0 END) as toplam_gelir,
            SUM(CASE WHEN h.tur = 'cikis' THEN h.tutar ELSE 0 END) as toplam_gider
        FROM hesap_hareketleri h
        WHERE h.tarih BETWEEN :baslangic AND :bitis AND h.sirket_id = :sirket_id
    ");
    $stmt_gelir_gider->execute([':baslangic' => $baslangic_tarihi, ':bitis' => $bitis_tarihi, ':sirket_id' => $aktif_sirket_id]);
    $gelir_gider = $stmt_gelir_gider->fetch(PDO::FETCH_ASSOC);
    
    // Hesap bazında bakiye raporu
    $stmt_hesap = $db->prepare("
        SELECT h.hesap_adi, h.bakiye, h.hesap_turu
        FROM hesaplar h
        WHERE h.sirket_id = :sirket_id
        ORDER BY h.bakiye DESC
    ");
    $stmt_hesap->execute([':sirket_id' => $aktif_sirket_id]);
    $hesap_bakiyeleri = $stmt_hesap->fetchAll(PDO::FETCH_ASSOC);
    
    // Vadesi geçmiş borç/alacak raporu
    $bugun = date('Y-m-d');
    $stmt_vadesi_gecmis = $db->prepare("
        SELECT 
            b.id, b.tur, b.tutar, b.tarih, b.vade_tarihi, b.aciklama,
            c.ad_soyad as cari_adi
        FROM borclar b
        JOIN cariler c ON b.cari_id = c.id
        WHERE b.vade_tarihi < :bugun AND b.durum != 'odendi' AND b.sirket_id = :sirket_id
        ORDER BY b.vade_tarihi ASC
    ");
    $stmt_vadesi_gecmis->execute([':bugun' => $bugun, ':sirket_id' => $aktif_sirket_id]);
    $vadesi_gecmis = $stmt_vadesi_gecmis->fetchAll(PDO::FETCH_ASSOC);
    
    // ŞİRKET DURUMU RAPORU
    // 1. Kasa Durumu (TABLO YAPISINA UYGUN GÜNCELLENDİ)
    $stmt_kasa = $db->prepare("
        SELECT 
            COUNT(*) as toplam_hesap,
            SUM(bakiye) as toplam_bakiye,
            SUM(CASE WHEN bakiye > 0 THEN bakiye ELSE 0 END) as pozitif_bakiye,
            SUM(CASE WHEN bakiye < 0 THEN bakiye ELSE 0 END) as negatif_bakiye
        FROM hesaplar 
        WHERE sirket_id = :sirket_id AND aktif = 1 AND hesap_turu IN ('kasa', 'banka')
    ");
    $stmt_kasa->execute([':sirket_id' => $aktif_sirket_id]);
    $kasa_durumu = $stmt_kasa->fetch(PDO::FETCH_ASSOC);
    
    // 2. Stok Durumu (Detaylı Stok Analizi)
    $stok_durumu = null;
    $stok_detay = null;
    $stok_islemleri_aktif = get_site_setting($db, 'stok_islemleri_aktif', '0');
    
    // Döviz kurlarını veritabanından al
    $doviz_kurlari = [];
    try {
        $stmt_doviz = $db->query("SELECT para_birimi, satis FROM doviz_kurlari ORDER BY guncelleme_tarihi DESC");
        while ($row = $stmt_doviz->fetch(PDO::FETCH_ASSOC)) {
            $doviz_kurlari[$row['para_birimi']] = (float)$row['satis'];
        }
    } catch (Exception $e) {
        error_log("Döviz kuru okuma hatası: " . $e->getMessage());
        // Hata durumunda boş array kullan
    }
    
    try {
        // Stok modülü aktif olup olmadığına bakılmaksızın stok bilgilerini al
        $stmt_stok_detay = $db->prepare("
            SELECT 
                COUNT(DISTINCT u.id) as toplam_urun_cesidi,
                SUM(COALESCE(ms.miktar, 0)) as toplam_stok_miktari,
                SUM(CASE WHEN COALESCE(ms.miktar, 0) > 0 THEN COALESCE(ms.miktar, 0) * u.alis_fiyati ELSE 0 END) as stok_alis_degeri,
                SUM(CASE WHEN COALESCE(ms.miktar, 0) > 0 THEN COALESCE(ms.miktar, 0) * u.satis_fiyati ELSE 0 END) as stok_satis_degeri,
                COUNT(CASE WHEN COALESCE(ms.miktar, 0) <= 0 THEN 1 END) as stoksuz_urun_sayisi
            FROM urunler u 
            LEFT JOIN mevcut_stok ms ON u.id = ms.urun_id 
            WHERE u.sirket_id = :sirket_id AND u.aktif = 1
        ");
        $stmt_stok_detay->execute([':sirket_id' => $aktif_sirket_id]);
        $stok_detay = $stmt_stok_detay->fetch(PDO::FETCH_ASSOC);
        
        // Çoklu para birimi için TL karşılığını hesapla
        $stok_alis_degeri_tl = 0;
        $stok_satis_degeri_tl = 0;
        
        // Stok işlemleri aktif olup olmadığına bakmaksızın hesapla
        $stmt_stok_tl = $db->prepare("
            SELECT 
                u.alis_fiyati, u.satis_fiyati, u.alis_para_birimi, u.satis_para_birimi,
                COALESCE(ms.miktar, 0) as stok_miktari
            FROM urunler u 
            LEFT JOIN mevcut_stok ms ON u.id = ms.urun_id 
            WHERE u.sirket_id = :sirket_id AND u.aktif = 1 AND COALESCE(ms.miktar, 0) > 0
        ");
        $stmt_stok_tl->execute([':sirket_id' => $aktif_sirket_id]);
        $stok_urunler = $stmt_stok_tl->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($stok_urunler as $urun) {
            $miktar = (float)$urun['stok_miktari'];
            $alis_fiyati = (float)$urun['alis_fiyati'];
            $satis_fiyati = (float)$urun['satis_fiyati'];
            $alis_para_birimi = $urun['alis_para_birimi'] ?? 'TL';
            $satis_para_birimi = $urun['satis_para_birimi'] ?? 'TL';
            
            // Alış değeri TL karşılığı
            $alis_kur = $doviz_kurlari[$alis_para_birimi] ?? 1.0;
            $stok_alis_degeri_tl += $alis_fiyati * $miktar * $alis_kur;
            
            // Satış değeri TL karşılığı
            $satis_kur = $doviz_kurlari[$satis_para_birimi] ?? 1.0;
            $stok_satis_degeri_tl += $satis_fiyati * $miktar * $satis_kur;
        }
        
        // Geriye uyumluluk için eski format + TL karşılığı
        $stok_durumu = [
            'stok_alis_degeri' => $stok_detay['stok_alis_degeri'] ?? 0,
            'stok_alis_degeri_tl' => $stok_alis_degeri_tl,
            'stok_satis_degeri_tl' => $stok_satis_degeri_tl
        ];
        
    } catch (PDOException $e) {
        error_log("Stok durumu hesaplama hatası: " . $e->getMessage());
        // Hata durumunda varsayılan değerler
        $stok_durumu = ['stok_alis_degeri' => 0, 'stok_alis_degeri_tl' => 0, 'stok_satis_degeri_tl' => 0];
        $stok_detay = [
            'toplam_urun_cesidi' => 0,
            'toplam_stok_miktari' => 0,
            'stok_alis_degeri' => 0,
            'stok_satis_degeri' => 0,
            'stoksuz_urun_sayisi' => 0
        ];
    }
    
    // 3. Cari Durumu (HESAPLAMA KESİN OLARAK DÜZELTİLDİ + DETAYLI ANALİZ)
    // Müşterilerden alacaklarımız (Müşterinin Borcu) - SADECE POZİTİF DEĞERLER
    $stmt_alacaklar = $db->prepare("SELECT SUM(CASE WHEN toplam_borc > 0 THEN toplam_borc ELSE 0 END) as toplam FROM cariler WHERE sirket_id = :sirket_id AND aktif = 1");
    $stmt_alacaklar->execute([':sirket_id' => $aktif_sirket_id]);
    $musteri_alacaklari = $stmt_alacaklar->fetchColumn() ?: 0;

    // Bizim tedarikçilere borçlarımız (Bizim Borcumuz)
    // Hem toplam_alacak kolonundaki pozitif değerler hem de toplam_borc kolonundaki negatif değerler
    $stmt_borclar = $db->prepare("
        SELECT 
            SUM(toplam_alacak) + SUM(CASE WHEN toplam_borc < 0 THEN ABS(toplam_borc) ELSE 0 END) as toplam 
        FROM cariler 
        WHERE sirket_id = :sirket_id
    ");
    $stmt_borclar->execute([':sirket_id' => $aktif_sirket_id]);
    $tedarikci_borclari = $stmt_borclar->fetchColumn() ?: 0;
    
    // Cari detay analizi - kişi sayıları
    try {
        $stmt_cari_detay = $db->prepare("
            SELECT 
                COUNT(CASE WHEN toplam_borc > 0 THEN 1 END) as alacakli_kisi_sayisi,
                COUNT(CASE WHEN toplam_alacak > 0 OR toplam_borc < 0 THEN 1 END) as borclu_kisi_sayisi,
                COUNT(CASE WHEN toplam_borc > 0 AND toplam_alacak > 0 THEN 1 END) as karmasik_durum_sayisi,
                COUNT(*) as toplam_cari_sayisi,
                SUM(CASE WHEN toplam_borc > 0 THEN toplam_borc END) as toplam_alacak,
                SUM(CASE WHEN toplam_alacak > 0 OR toplam_borc < 0 THEN 
                    CASE 
                        WHEN toplam_alacak > 0 AND toplam_borc < 0 THEN toplam_alacak + ABS(toplam_borc)
                        WHEN toplam_alacak > 0 THEN toplam_alacak
                        WHEN toplam_borc < 0 THEN ABS(toplam_borc)
                        ELSE 0
                    END
                END) as toplam_borc
            FROM cariler 
            WHERE sirket_id = :sirket_id AND aktif = 1
        ");
        $stmt_cari_detay->execute([':sirket_id' => $aktif_sirket_id]);
        $cari_detay = $stmt_cari_detay->fetch(PDO::FETCH_ASSOC);
        
    } catch (PDOException $e) {
        error_log("Cari detay analizi hatası: " . $e->getMessage());
        // Hata durumunda varsayılan değerler
        $cari_detay = [
            'alacakli_kisi_sayisi' => 0,
            'borclu_kisi_sayisi' => 0,
            'karmasik_durum_sayisi' => 0,
            'toplam_cari_sayisi' => 0,
            'toplam_alacak' => 0,
            'toplam_borc' => 0
        ];
    }

    // 4. Genel Mali Durum Hesaplama (HESAPLAMA DÜZELTİLDİ)
    $toplam_varlik = ($kasa_durumu['toplam_bakiye'] ?: 0) + 
                     ($stok_durumu ? ($stok_durumu['stok_alis_degeri_tl'] ?: 0) : 0) + 
                     $musteri_alacaklari;
                     
    $diger_yukumlulukler_toplami = $diger_yukumlulukler_toplami ?? 0;
    $toplam_yükümlülük = $tedarikci_borclari + abs($diger_yukumlulukler_toplami);
    
    $net_servet = $toplam_varlik - $toplam_yükümlülük;

    // 5. Kasa ve Banka Hesap Detayları (TABLO YAPISINA UYGUN GÜNCELLENDİ)
    $stmt_hesaplar = $db->prepare("SELECT id, hesap_adi, hesap_turu, bakiye FROM hesaplar WHERE sirket_id = :sirket_id AND aktif = 1");
    $stmt_hesaplar->execute([':sirket_id' => $aktif_sirket_id]);
    $hesaplar_detay_raw = $stmt_hesaplar->fetchAll(PDO::FETCH_ASSOC);

    $varlik_hesaplari = [];
    $yukumluluk_hesaplari = [];
    $diger_yukumlulukler_toplami = 0;

    foreach ($hesaplar_detay_raw as $hesap) {
        // hesap_turu'ne göre sınıflandırma
        if ($hesap['hesap_turu'] === 'kredi_karti') {
            // Kredi kartları her zaman yükümlülük olarak kabul edilir
            $yukumluluk_hesaplari[] = $hesap;
            $diger_yukumlulukler_toplami += $hesap['bakiye'];
        } else {
            // Kasa ve banka hesapları varlık olarak kabul edilir
            $varlik_hesaplari[] = $hesap;
        }
    }
    
    // Karlılık analizi (son 3 aylık)
    $uc_ay_once = date('Y-m-d', strtotime('-3 months'));
    $stmt_karlilik = $db->prepare("
        SELECT 
            SUM(CASE WHEN tur = 'giris' THEN tutar ELSE 0 END) as son_3_ay_gelir,
            SUM(CASE WHEN tur = 'cikis' THEN tutar ELSE 0 END) as son_3_ay_gider
        FROM hesap_hareketleri 
        WHERE sirket_id = :sirket_id AND tarih >= :uc_ay_once
    ");
    $stmt_karlilik->execute([':sirket_id' => $aktif_sirket_id, ':uc_ay_once' => $uc_ay_once]);
    $karlilik = $stmt_karlilik->fetch(PDO::FETCH_ASSOC);
    
    $son_3_ay_kar = ($karlilik['son_3_ay_gelir'] ?: 0) - ($karlilik['son_3_ay_gider'] ?: 0);

    // Stok Değer Raporu için ürün listesi
    $stok_deger_raporu = [];
    if ($stok_islemleri_aktif == '1') {
        $stmt_stok_deger = $db->prepare("
            SELECT
                u.urun_adi,
                u.alis_fiyati,
                u.alis_para_birimi,
                u.satis_fiyati,
                u.satis_para_birimi,
                COALESCE(ms.miktar, 0) as stok_miktari,
                (COALESCE(ms.miktar, 0) * u.alis_fiyati) as toplam_deger
            FROM urunler u
            LEFT JOIN mevcut_stok ms ON u.id = ms.urun_id
            WHERE u.sirket_id = :sirket_id AND u.aktif = 1 AND COALESCE(ms.miktar, 0) > 0
            ORDER BY (COALESCE(ms.miktar, 0) * u.alis_fiyati) DESC
        ");
        $stmt_stok_deger->execute([':sirket_id' => $aktif_sirket_id]);
        $stok_deger_raporu_raw = $stmt_stok_deger->fetchAll(PDO::FETCH_ASSOC);
        
        // TL karşılıklarını hesapla
        foreach ($stok_deger_raporu_raw as $urun) {
            $alis_para_birimi = $urun['alis_para_birimi'] ?? 'TL';
            $alis_kur = $doviz_kurlari[$alis_para_birimi] ?? 1.0;
            $toplam_deger_tl = $urun['toplam_deger'] * $alis_kur;
            
            $stok_deger_raporu[] = [
                'urun_adi' => $urun['urun_adi'],
                'stok_miktari' => $urun['stok_miktari'],
                'alis_fiyati' => $urun['alis_fiyati'],
                'alis_para_birimi' => $alis_para_birimi,
                'toplam_deger' => $urun['toplam_deger'],
                'toplam_deger_tl' => $toplam_deger_tl
            ];
        }
    }
    
} catch(PDOException $e) {
    error_log("Rapor hatası: " . $e->getMessage());
    die(__("an_error_occurred_while_retrieving_reports_please_try_again_later"));
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= __("odjoo_management") ?> - <?= htmlspecialchars($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container-xl mt-4">
        <div class="row mb-4">
            <div class="col-md-12">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h2 class="mb-0"><i class="bi bi-graph-up"></i> <?= htmlspecialchars($sayfa_baslik) ?></h2>
                    <a href="pdf_export.php?type=raporlar&baslangic_tarihi=<?= htmlspecialchars($baslangic_tarihi) ?>&bitis_tarihi=<?= htmlspecialchars($bitis_tarihi) ?>" class="btn btn-sm btn-outline-primary">
                        <i class="bi bi-file-pdf"></i> PDF İndir
                    </a>
                </div>

                <!-- Tarih Aralığı Filtreleme -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-calendar-range"></i> <?= __("date_range_filtering") ?></h5>
                    </div>
                    <div class="card-body">
                        <form method="get" class="row g-3">
                            <div class="col-md-4">
                                <label for="baslangic_tarihi" class="form-label"><?= __("start_date") ?></label>
                                <input type="date" class="form-control" id="baslangic_tarihi" name="baslangic_tarihi" 
                                       value="<?= htmlspecialchars($baslangic_tarihi) ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="bitis_tarihi" class="form-label"><?= __("end_date") ?></label>
                                <input type="date" class="form-control" id="bitis_tarihi" name="bitis_tarihi" 
                                       value="<?= htmlspecialchars($bitis_tarihi) ?>">
                            </div>
                            <div class="col-md-4 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-filter"></i> <?= __("filter") ?>
                                </button>
                                <a href="raporlar.php" class="btn btn-outline-secondary ms-2">
                                    <i class="bi bi-x-circle"></i> <?= __("clear_filter") ?>
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="row">
            <div class="col-md-7 mb-4">
                <div class="card shadow">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-cash-stack"></i> <?= __("income_expense_summary") ?> (<?= date('d.m.Y', strtotime($baslangic_tarihi)) ?> - <?= date('d.m.Y', strtotime($bitis_tarihi)) ?>)</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card bg-success text-white">
                                    <div class="card-body">
                                        <h5 class="card-title"><?= __("total_income") ?></h5>
                                        <h3><?= number_format($gelir_gider['toplam_gelir'] ?? 0, 2, ',', '.') ?> <?= $paraBirimi ?></h3>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card bg-danger text-white">
                                    <div class="card-body">
                                        <h5 class="card-title"><?= __("total_expense") ?></h5>
                                        <h3><?= number_format($gelir_gider['toplam_gider'] ?? 0, 2, ',', '.') ?> <?= $paraBirimi ?></h3>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card <?= (($gelir_gider['toplam_gelir'] ?? 0) - ($gelir_gider['toplam_gider'] ?? 0)) >= 0 ? 'bg-primary' : 'bg-warning text-dark' ?> text-white">
                                    <div class="card-body">
                                        <h5 class="card-title"><?= __("net_status") ?></h5>
                                        <h3><?= number_format(($gelir_gider['toplam_gelir'] ?? 0) - ($gelir_gider['toplam_gider'] ?? 0), 2, ',', '.') ?> <?= $paraBirimi ?></h3>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-5 mb-4">
                <div class="card shadow">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-pie-chart"></i> <?= __("debts_by_category") ?></h5>
                    </div>
                    <div class="card-body">
                        <canvas id="kategoriChart" height="250"></canvas>
                    </div>
                </div>
                
            </div>
        </div>
        
        <!-- Hesap Bakiyeleri -->
        <div class="row">
            <div class="col-md-12 mb-4">
                <div class="card shadow">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-bank"></i> <?= __("account_balances") ?></h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th><?= __("account_name") ?></th>
                                        <th><?= __("account_type") ?></th>
                                        <th><?= __("balance") ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($hesap_bakiyeleri as $hesap): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($hesap['hesap_adi']) ?></td>
                                        <td>
                                            <?php 
                                            $hesap_turu_text = "Diğer";
                                            switch($hesap['hesap_turu']) {
                                                case 'nakit': $hesap_turu_text = "Nakit"; break;
                                                case 'banka': $hesap_turu_text = "Banka"; break;
                                                case 'kredi_karti': $hesap_turu_text = "Kredi Kartı"; break;
                                            }
                                            echo htmlspecialchars($hesap_turu_text);
                                            ?>
                                        </td>
                                        <td class="<?= $hesap['bakiye'] >= 0 ? 'text-success' : 'text-danger' ?>">
                                            <?= number_format($hesap['bakiye'], 2, ',', '.') ?> <?= $paraBirimi ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot class="table-light">
                                    <tr>
                                        <th colspan="2"><?= __("total_balance") ?></th>
                                        <th class="<?= array_sum(array_column($hesap_bakiyeleri, 'bakiye')) >= 0 ? 'text-success' : 'text-danger' ?>">
                                            <?= number_format(array_sum(array_column($hesap_bakiyeleri, 'bakiye')), 2, ',', '.') ?> <?= $paraBirimi ?>
                                        </th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Vadesi Geçmiş Borçlar/Alacaklar -->
        <div class="row">
            <div class="col-md-12 mb-4">
                <div class="card shadow">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-exclamation-triangle"></i> <?= __("overdue_debts_credits") ?></h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($vadesi_gecmis)): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle"></i> <?= __("no_overdue_debts_or_credits_found") ?>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead class="table-light">
                                        <tr>
                                            <th><?= __("type") ?></th>
                                            <th><?= __("current") ?></th>
                                            <th><?= __("amount") ?></th>
                                            <th><?= __("due_date") ?></th>
                                            <th><?= __("delay") ?></th>
                                            <th><?= __("description") ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach($vadesi_gecmis as $vade): 
                                            $gecikme_gun = floor((strtotime($bugun) - strtotime($vade['vade_tarihi'])) / (60 * 60 * 24));
                                        ?>
                                        <tr>
                                            <td>
                                                <?php if ($vade['tur'] == 'satis'): ?>
                                                    <span class="badge bg-success"><?= __("receivable") ?></span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger"><?= __("debt") ?></span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <a href="cari_detay.php?id=<?= $vade['id'] ?>">
                                                    <?= htmlspecialchars($vade['cari_adi']) ?>
                                                </a>
                                            </td>
                                            <td><?= number_format($vade['tutar'], 2, ',', '.') ?> <?= $paraBirimi ?></td>
                                            <td><?= date('d.m.Y', strtotime($vade['vade_tarihi'])) ?></td>
                                            <td>
                                                <span class="badge bg-warning text-dark"><?= $gecikme_gun . ' ' . ($gecikme_gun == 1 ? __("day_late_singular") : __("days_late_plural")) ?></span>
                                            </td>
                                            <td><?= htmlspecialchars($vade['aciklama']) ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- DETAYLI FİNANSAL DURUM RAPORU -->
        <div class="row mt-4">
            <div class="col-12">
                <h4 class="mb-3"><i class="bi bi-file-earmark-medical me-2"></i><?= __("financial_status_summary") ?></h4>
            </div>

            <!-- VARLIKLAR -->
            <div class="col-lg-4 mb-4">
                <div class="card shadow-sm h-100 border-success">
                    <div class="card-header bg-success text-white text-center">
                        <h5 class="mb-0"><i class="bi bi-plus-circle me-2"></i><?= strtoupper(__("assets")) ?></h5>
                    </div>
                    <div class="card-body p-3">
                        <!-- Kasa ve Banka -->
                        <div class="d-flex justify-content-between align-items-center mb-3 p-2 bg-light rounded">
                            <div>
                                <i class="bi bi-cash-stack text-primary me-2"></i>
                                <strong><?= __("cash_bank") ?></strong>
                            </div>
                            <span class="badge bg-primary fs-6"><?= number_format($kasa_durumu['toplam_bakiye'], 2, ',', '.') ?> ₺</span>
                        </div>
                        
                        <!-- Ticari Alacaklar -->
                        <div class="d-flex justify-content-between align-items-center mb-3 p-2 bg-light rounded">
                            <div>
                                <i class="bi bi-person-check text-success me-2"></i>
                                <strong><?= __("customer_receivables") ?></strong>
                            </div>
                            <span class="badge bg-success fs-6"><?= number_format($musteri_alacaklari, 2, ',', '.') ?> ₺</span>
                        </div>
                        
                        <!-- Stok Değeri -->
                        <div class="d-flex justify-content-between align-items-center mb-3 p-2 bg-light rounded">
                            <div>
                                <i class="bi bi-box-seam text-info me-2"></i>
                                <strong><?= __("stock_value") ?></strong>
                            </div>
                            <span class="badge bg-info fs-6"><?= number_format($stok_durumu['stok_alis_degeri_tl'] ?? 0, 2, ',', '.') ?> ₺</span>
                        </div>
                        
                        <hr class="my-3">
                        
                        <!-- Toplam Varlıklar -->
                        <div class="d-flex justify-content-between align-items-center p-3 bg-success-subtle rounded">
                            <h6 class="mb-0 text-success"><strong><?= strtoupper(__("total_assets")) ?></strong></h6>
                            <h5 class="mb-0 text-success"><?= number_format($toplam_varlik, 2, ',', '.') ?> ₺</h5>
                        </div>
                    </div>
                </div>
            </div>

            <!-- YÜKÜMLÜLÜKLER -->
            <div class="col-lg-4 mb-4">
                <div class="card shadow-sm h-100 border-danger">
                    <div class="card-header bg-danger text-white text-center">
                        <h5 class="mb-0"><i class="bi bi-dash-circle me-2"></i><?= strtoupper(__("liabilities")) ?></h5>
                    </div>
                    <div class="card-body p-3">
                        <!-- Ticari Borçlar -->
                        <div class="d-flex justify-content-between align-items-center mb-3 p-2 bg-light rounded">
                            <div>
                                <i class="bi bi-person-x text-danger me-2"></i>
                                <strong><?= __("supplier_debts") ?></strong>
                            </div>
                            <span class="badge bg-danger fs-6"><?= number_format($tedarikci_borclari, 2, ',', '.') ?> ₺</span>
                        </div>
                        
                        <!-- Kredi Kartı Borçları -->
                        <?php if (!empty($yukumluluk_hesaplari)): ?>
                        <div class="d-flex justify-content-between align-items-center mb-3 p-2 bg-light rounded">
                            <div>
                                <i class="bi bi-credit-card-2-front text-warning me-2"></i>
                                <strong><?= __("credit_card") ?></strong>
                            </div>
                            <span class="badge bg-warning text-dark fs-6"><?= number_format($diger_yukumlulukler_toplami, 2, ',', '.') ?> ₺</span>
                        </div>
                        <?php endif; ?>
                        
                        <hr class="my-3">
                        
                        <!-- Toplam Yükümlülükler -->
                        <div class="d-flex justify-content-between align-items-center p-3 bg-danger-subtle rounded">
                            <h6 class="mb-0 text-danger"><strong><?= strtoupper(__("total_liabilities")) ?></strong></h6>
                            <h5 class="mb-0 text-danger"><?= number_format($toplam_yükümlülük, 2, ',', '.') ?> ₺</h5>
                        </div>
                    </div>
                </div>
            </div>

            <!-- NET DEĞER -->
            <div class="col-lg-4 mb-4">
                <div class="card shadow-sm h-100 border-primary">
                    <div class="card-header bg-primary text-white text-center">
                        <h5 class="mb-0"><i class="bi bi-gem me-2"></i><?= strtoupper(__("net_worth")) ?></h5>
                    </div>
                    <div class="card-body p-3 text-center">
                        <!-- Net Değer -->
                        <div class="mb-3">
                            <h6 class="text-muted"><?= __("assets_liabilities") ?></h6>
                            <h2 class="my-2 fw-bold <?= $net_servet >= 0 ? 'text-success' : 'text-danger' ?>">
                                <?= number_format($net_servet, 2, ',', '.') ?> ₺
                            </h2>
                        </div>
                        
                        <!-- Finansal Sağlık -->
                        <div class="mb-3">
                            <?php 
                            $finansal_durum = '';
                            $durum_class = '';
                            $durum_icon = '';
                            
                            if ($net_servet >= $toplam_varlik * 0.5) {
                                $finansal_durum = 'Mükemmel';
                                $durum_class = 'success';
                                $durum_icon = 'bi-check-circle-fill';
                            } elseif ($net_servet >= $toplam_varlik * 0.3) {
                                $finansal_durum = 'İyi';
                                $durum_class = 'success';
                                $durum_icon = 'bi-check-circle';
                            } elseif ($net_servet >= $toplam_varlik * 0.1) {
                                $finansal_durum = 'Orta';
                                $durum_class = 'warning';
                                $durum_icon = 'bi-exclamation-triangle';
                            } elseif ($net_servet >= 0) {
                                $finansal_durum = 'Zayıf';
                                $durum_class = 'warning';
                                $durum_icon = 'bi-exclamation-triangle-fill';
                            } else {
                                $finansal_durum = 'Kritik';
                                $durum_class = 'danger';
                                $durum_icon = 'bi-x-circle-fill';
                            }
                            ?>
                            <div class="alert alert-<?= $durum_class ?> py-2 mb-0">
                                <i class="bi <?= $durum_icon ?> me-2"></i>
                                <strong><?= __("financial_status") ?>: <?= $finansal_durum ?></strong>
                            </div>
                        </div>
                        
                        <!-- Oranlar -->
                        <div class="row text-center">
                            <div class="col-6">
                                <small class="text-muted"><?= __("assets_liabilities") ?></small>
                                <div class="fw-bold text-primary"><?= number_format($toplam_varlik / max($toplam_yükümlülük, 1), 2) ?></div>
                            </div>
                            <div class="col-6">
                                <small class="text-muted"><?= __("equity_percentage") ?></small>
                                <div class="fw-bold text-primary">%<?= number_format(($net_servet / max($toplam_varlik, 1)) * 100, 1) ?></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- NET CARİ POZİSYON - Ayrı Kart -->
        <div class="row mt-2">
            <div class="col-12">
                <div class="card shadow-sm border-warning">
                    <div class="card-header bg-warning text-dark text-center">
                        <h5 class="mb-0"><i class="bi bi-arrow-left-right me-2"></i><?= strtoupper(__("net_current_position")) ?></h5>
                    </div>
                    <div class="card-body p-4">
                        <div class="row text-center">
                            <div class="col-md-4">
                                <div class="p-3 bg-success-subtle rounded">
                                    <i class="bi bi-arrow-up-circle text-success fs-1"></i>
                                    <h6 class="text-success mt-2"><?= __("our_total_receivables") ?></h6>
                                    <h4 class="fw-bold text-success"><?= number_format($musteri_alacaklari, 2, ',', '.') ?> ₺</h4>
                                    <small class="text-muted"><?= __("receivables_from_customers") ?></small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="p-3 bg-danger-subtle rounded">
                                    <i class="bi bi-arrow-down-circle text-danger fs-1"></i>
                                    <h6 class="text-danger mt-2"><?= __("our_total_debts") ?></h6>
                                    <h4 class="fw-bold text-danger"><?= number_format($tedarikci_borclari, 2, ',', '.') ?> ₺</h4>
                                    <small class="text-muted"><?= __("debts_to_suppliers") ?></small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="p-3 <?= ($musteri_alacaklari - $tedarikci_borclari) >= 0 ? 'bg-success-subtle' : 'bg-danger-subtle' ?> rounded">
                                    <i class="bi <?= ($musteri_alacaklari - $tedarikci_borclari) >= 0 ? 'bi-check-circle-fill text-success' : 'bi-x-circle-fill text-danger' ?> fs-1"></i>
                                    <h6 class="<?= ($musteri_alacaklari - $tedarikci_borclari) >= 0 ? 'text-success' : 'text-danger' ?> mt-2">Net Pozisyon</h6>
                                    <h4 class="fw-bold <?= ($musteri_alacaklari - $tedarikci_borclari) >= 0 ? 'text-success' : 'text-danger' ?>">
                                        <?= number_format($musteri_alacaklari - $tedarikci_borclari, 2, ',', '.') ?> ₺
                                    </h4>
                                    <small class="text-muted">
                                        <?php if (($musteri_alacaklari - $tedarikci_borclari) >= 0): ?>
                                            <?= __("our_receivables_exceed_debts") ?>
                                        <?php else: ?>
                                            <?= __("our_debts_exceed_receivables") ?>
                                        <?php endif; ?>
                                    </small>

                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <?php if ($stok_islemleri_aktif == '1' && !empty($stok_deger_raporu)): ?>
        <!-- Stok Değer Raporu -->
        <div class="row mt-4">
            <div class="col-md-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0"><i class="bi bi-boxes"></i> <?= __("stock_value_report") ?> (<?= __("in_tl") ?>)</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th><?= __("product_name") ?></th>
                                        <th class="text-end"><?= __("stock_quantity") ?></th>
                                        <th class="text-end"><?= __("unit_purchase_price") ?></th>
                                        <th class="text-end"><?= __("original_value") ?></th>
                                        <th class="text-end"><?= __("equivalent_in_tl") ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($stok_deger_raporu as $urun): ?>
                                    <tr>
                                        <td class="text-end"><?= number_format($urun['stok_miktari']) ?></td>
                                        <td class="text-end"><?= number_format($urun['alis_fiyati'], 2) ?> <?= $urun['alis_para_birimi'] ?></td>
                                        <td class="text-end fw-bold"><?= number_format($urun['toplam_deger'], 2) ?> <?= $urun['alis_para_birimi'] ?></td>
                                        <td class="text-end fw-bold"><?= number_format($urun['toplam_deger_tl'], 2) ?> TL</td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot class="table-dark">
                                    <tr>
                                        <th colspan="3" class="text-end"><?= __("total_stock_value_original") ?></th>
                                        <th class="text-end fs-5"><?= number_format($stok_durumu['stok_alis_degeri'] ?: 0, 2) ?> (<?= __("multiple_currencies") ?>)</th>
                                        <th class="text-end fs-5"><?= number_format($stok_durumu['stok_alis_degeri_tl'] ?: 0, 2) ?> TL</th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Debug Bilgileri (Geliştirici için) -->
        <?php if (isset($_GET['debug']) && $_GET['debug'] == '1'): ?>
        <div class="row mt-4">
            <div class="col-md-12">
                <div class="card border-info">
                    <div class="card-header bg-info text-white">
                        <h6 class="mb-0"><i class="bi bi-bug"></i> <?= __("debug_information") ?></h6>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6><?= __("exchange_rates") ?>:</h6>
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr><th><?= __("currency") ?></th><th><?= __("rate") ?></th></tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($doviz_kurlari as $para_birimi => $kur): ?>
                                            <tr>
                                                <td><?= $para_birimi ?></td>
                                                <td><?= number_format($kur, 4) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6><?= __("stock_information") ?>:</h6>
                                <ul class="list-unstyled">
                                    <li><strong><?= __("stock_product_count") ?>:</strong> <?= count($stok_urunler ?? []) ?></li>
                                    <li><strong><?= __("total_product_variety") ?>:</strong> <?= $stok_detay['toplam_urun_cesidi'] ?? 0 ?></li>
                                    <li><strong><?= __("total_stock_quantity") ?>:</strong> <?= number_format($stok_detay['toplam_stok_miktari'] ?? 0, 2) ?></li>
                                    <li><strong>Orijinal Stok Değeri:</strong> <?= number_format($stok_durumu['stok_alis_degeri'] ?? 0, 2) ?></li>
                                    <li><strong>TL Karşılığı:</strong> <?= number_format($stok_durumu['stok_alis_degeri_tl'] ?? 0, 2) ?> TL</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // Kategori Chart
        const kategoriCtx = document.getElementById('kategoriChart').getContext('2d');
        const kategoriData = <?= json_encode($kategori_borclari) ?>;
        new Chart(kategoriCtx, {
            type: 'doughnut',
            data: {
                labels: kategoriData.map(item => item.kategori_adi),
                datasets: [{
                    label: 'Toplam Borç',
                    data: kategoriData.map(item => item.toplam_borc),
                    backgroundColor: [
                        'rgba(255, 99, 132, 0.8)', 'rgba(54, 162, 235, 0.8)',
                        'rgba(255, 206, 86, 0.8)', 'rgba(75, 192, 192, 0.8)',
                        'rgba(153, 102, 255, 0.8)', 'rgba(255, 159, 64, 0.8)'
                    ],
                    borderColor: '#fff',
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    title: {
                        display: false,
                        text: 'Kategori Bazında Borç Dağılımı'
                    }
                }
            }
        });
    });
    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <?php include __DIR__ . '/../includes/footer.php'; ?>
</body>
</html>
