<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi();

// TCPDF kütüphanesini dahil et
require_once('tcpdf/tcpdf.php');

// PDF Türü ve ID parametrelerini al
$pdf_type = isset($_GET['type']) ? $_GET['type'] : '';
$id = isset($_GET['id']) ? filter_var($_GET['id'], FILTER_VALIDATE_INT) : 0;

// Tarih aralığı parametrelerini al (raporlar için)
$baslangic_tarihi = isset($_GET['baslangic_tarihi']) ? $_GET['baslangic_tarihi'] : date('Y-m-01');
$bitis_tarihi = isset($_GET['bitis_tarihi']) ? $_GET['bitis_tarihi'] : date('Y-m-d');

// Geçerli parametreler yoksa hata ver
if (empty($pdf_type) || ($pdf_type != 'rapor' && empty($id))) {
    die("Geçersiz PDF parametreleri.");
}

// TCPDF sınıfını genişlet (Türkçe karakter desteği için)
class MYPDF extends TCPDF {
    public function Header() {
        $this->SetFont('dejavusans', 'B', 12);
        $this->Cell(0, 10, 'Odjoo Yönetim Sistemi', 0, false, 'C', 0, '', 0, false, 'M', 'M');
        $this->Ln(15);
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('dejavusans', 'I', 8);
        $this->Cell(0, 10, 'Sayfa '.$this->getAliasNumPage().'/'.$this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
        $this->Cell(0, 10, 'Oluşturulma Tarihi: ' . date('d.m.Y H:i'), 0, false, 'R', 0, '', 0, false, 'T', 'M');
    }
}

// Yeni PDF dokümanı oluştur
$pdf = new MYPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

// Doküman bilgilerini ayarla
$pdf->SetCreator('Odjoo Yönetim Sistemi');
$pdf->SetAuthor('Odjoo Yönetim');
$pdf->SetTitle('Odjoo Yönetim Raporu');
$pdf->SetSubject('Odjoo Yönetim Raporu');

// Varsayılan başlık ve alt bilgi ayarla
$pdf->setHeaderFont(Array('dejavusans', 'B', 10));
$pdf->setFooterFont(Array('dejavusans', 'I', 8));

// Varsayılan monospace yazı tipi
$pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

// Marjinleri ayarla
$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
$pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
$pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

// Otomatik sayfa sonu ayarla
$pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

// Görüntü ölçekleme faktörünü ayarla
$pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);

// Yazı tipi ayarla
$pdf->SetFont('dejavusans', '', 10);

// Yeni sayfa ekle
$pdf->AddPage();

// İçerik oluştur
switch ($pdf_type) {
    case 'cari_ekstre':
        generateCariEkstre($pdf, $id, $db);
        break;
    case 'hesap_hareket':
        generateHesapHareket($pdf, $id, $db);
        break;
    case 'rapor':
        generateRapor($pdf, $baslangic_tarihi, $bitis_tarihi, $db);
        break;
    default:
        $pdf->Cell(0, 10, 'Geçersiz PDF türü.', 0, 1, 'C');
        break;
}

// PDF'i gönder
$pdf->Output('Odjoo_rapor_'.date('YmdHis').'.pdf', 'I');
exit;

// Cari Ekstre PDF'i oluştur
function generateCariEkstre($pdf, $cari_id, $db) {
    try {
        // Cari bilgilerini al
        $stmt_cari = $db->prepare("SELECT * FROM cariler WHERE id = ?");
        $stmt_cari->execute([$cari_id]);
        $cari = $stmt_cari->fetch(PDO::FETCH_ASSOC);
        
        if (!$cari) {
            $pdf->Cell(0, 10, 'Cari bulunamadı.', 0, 1, 'C');
            return;
        }
        
        // Cari borç/alacak hareketlerini al
        $stmt_borclar = $db->prepare("
            SELECT b.*, k.kategori_adi 
            FROM borclar b
            LEFT JOIN kategoriler k ON b.kategori_id = k.id
            WHERE b.cari_id = ?
            ORDER BY b.tarih DESC
        ");
        $stmt_borclar->execute([$cari_id]);
        $borclar = $stmt_borclar->fetchAll(PDO::FETCH_ASSOC);
        
        // Cari ödemeleri al
        $stmt_odemeler = $db->prepare("
            SELECT hh.*, h.hesap_adi 
            FROM hesap_hareketleri hh
            LEFT JOIN hesaplar h ON hh.hesap_id = h.id
            WHERE hh.cari_id = ?
            ORDER BY hh.tarih DESC
        ");
        $stmt_odemeler->execute([$cari_id]);
        $odemeler = $stmt_odemeler->fetchAll(PDO::FETCH_ASSOC);
        
        // PDF başlığı
        $pdf->SetFont('dejavusans', 'B', 16);
        $pdf->Cell(0, 10, 'Cari Ekstre: ' . $cari['ad_soyad'], 0, 1, 'C');
        $pdf->Ln(5);
        
        // Cari Bilgileri
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Cari Bilgileri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', '', 10);
        
        $pdf->Cell(40, 7, 'Cari Adı:', 0, 0, 'L');
        $pdf->Cell(0, 7, $cari['ad_soyad'], 0, 1, 'L');
        
        if (!empty($cari['telefon'])) {
            $pdf->Cell(40, 7, 'Telefon:', 0, 0, 'L');
            $pdf->Cell(0, 7, $cari['telefon'], 0, 1, 'L');
        }
        
        if (!empty($cari['eposta'])) {
            $pdf->Cell(40, 7, 'E-posta:', 0, 0, 'L');
            $pdf->Cell(0, 7, $cari['eposta'], 0, 1, 'L');
        }
        
        if (!empty($cari['adres'])) {
            $pdf->Cell(40, 7, 'Adres:', 0, 0, 'L');
            $pdf->MultiCell(0, 7, $cari['adres'], 0, 'L');
        }
        
        $pdf->Cell(40, 7, 'Toplam Borç:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($cari['toplam_borc'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $pdf->Cell(40, 7, 'Toplam Alacak:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($cari['toplam_alacak'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $pdf->Cell(40, 7, 'Bakiye:', 0, 0, 'L');
        $bakiye_text = number_format($cari['bakiye'], 2, ',', '.') . ' ' . getParaBirimiSembolu();
        $pdf->Cell(0, 7, $bakiye_text, 0, 1, 'L');
        
        $pdf->Ln(5);
        
        // Borç/Alacak Tablosu
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Borç/Alacak Hareketleri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(25, 7, 'Tarih', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Tür', 1, 0, 'C');
        $pdf->Cell(30, 7, 'Kategori', 1, 0, 'C');
        $pdf->Cell(70, 7, 'Açıklama', 1, 0, 'C');
        $pdf->Cell(20, 7, 'Vade', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Tutar', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        
        if (empty($borclar)) {
            $pdf->Cell(195, 7, 'Kayıt bulunamadı.', 1, 1, 'C');
        } else {
            foreach ($borclar as $borc) {
                $pdf->Cell(25, 7, date('d.m.Y', strtotime($borc['tarih'])), 1, 0, 'C');
                $pdf->Cell(25, 7, $borc['tur'] == 'satis' ? 'Satış' : 'Alış', 1, 0, 'C');
                $pdf->Cell(30, 7, $borc['kategori_adi'], 1, 0, 'L');
                $pdf->Cell(70, 7, mb_substr($borc['aciklama'], 0, 30), 1, 0, 'L');
                $pdf->Cell(20, 7, $borc['vade_tarihi'] ? date('d.m.Y', strtotime($borc['vade_tarihi'])) : '-', 1, 0, 'C');
                $pdf->Cell(25, 7, number_format($borc['tutar'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
            }
        }
        
        $pdf->Ln(5);
        
        // Ödeme Tablosu
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Ödeme Hareketleri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(25, 7, 'Tarih', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Tür', 1, 0, 'C');
        $pdf->Cell(40, 7, 'Hesap', 1, 0, 'C');
        $pdf->Cell(80, 7, 'Açıklama', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Tutar', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        
        if (empty($odemeler)) {
            $pdf->Cell(195, 7, 'Kayıt bulunamadı.', 1, 1, 'C');
        } else {
            foreach ($odemeler as $odeme) {
                $pdf->Cell(25, 7, date('d.m.Y', strtotime($odeme['tarih'])), 1, 0, 'C');
                $pdf->Cell(25, 7, $odeme['tur'] == 'giris' ? 'Tahsilat' : 'Ödeme', 1, 0, 'C');
                $pdf->Cell(40, 7, $odeme['hesap_adi'], 1, 0, 'L');
                $pdf->Cell(80, 7, mb_substr($odeme['aciklama'], 0, 40), 1, 0, 'L');
                $pdf->Cell(25, 7, number_format($odeme['tutar'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
            }
        }
        
    } catch (PDOException $e) {
        $pdf->Cell(0, 10, 'Veri çekme hatası: ' . $e->getMessage(), 0, 1, 'C');
    }
}

// Hesap Hareket PDF'i oluştur
function generateHesapHareket($pdf, $hesap_id, $db) {
    try {
        // Hesap bilgilerini al
        $stmt_hesap = $db->prepare("SELECT * FROM hesaplar WHERE id = ?");
        $stmt_hesap->execute([$hesap_id]);
        $hesap = $stmt_hesap->fetch(PDO::FETCH_ASSOC);
        
        if (!$hesap) {
            $pdf->Cell(0, 10, 'Hesap bulunamadı.', 0, 1, 'C');
            return;
        }
        
        // Hesap hareketlerini al
        $stmt_hareketler = $db->prepare("
            SELECT hh.*, c.ad_soyad as cari_adi
            FROM hesap_hareketleri hh
            LEFT JOIN cariler c ON hh.cari_id = c.id
            WHERE hh.hesap_id = ?
            ORDER BY hh.tarih DESC
        ");
        $stmt_hareketler->execute([$hesap_id]);
        $hareketler = $stmt_hareketler->fetchAll(PDO::FETCH_ASSOC);
        
        // PDF başlığı
        $pdf->SetFont('dejavusans', 'B', 16);
        $pdf->Cell(0, 10, 'Hesap Hareketleri: ' . $hesap['hesap_adi'], 0, 1, 'C');
        $pdf->Ln(5);
        
        // Hesap Bilgileri
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Hesap Bilgileri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', '', 10);
        
        $pdf->Cell(40, 7, 'Hesap Adı:', 0, 0, 'L');
        $pdf->Cell(0, 7, $hesap['hesap_adi'], 0, 1, 'L');
        
        $pdf->Cell(40, 7, 'Hesap Türü:', 0, 0, 'L');
        $hesap_turu = '';
        switch ($hesap['hesap_turu']) {
            case 'nakit': $hesap_turu = 'Nakit'; break;
            case 'banka': $hesap_turu = 'Banka'; break;
            case 'kredi_karti': $hesap_turu = 'Kredi Kartı'; break;
            default: $hesap_turu = $hesap['hesap_turu']; break;
        }
        $pdf->Cell(0, 7, $hesap_turu, 0, 1, 'L');
        
        $pdf->Cell(40, 7, 'Bakiye:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($hesap['bakiye'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $pdf->Ln(5);
        
        // Hareket Tablosu
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Hesap Hareketleri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(25, 7, 'Tarih', 1, 0, 'C');
        $pdf->Cell(20, 7, 'Tür', 1, 0, 'C');
        $pdf->Cell(40, 7, 'Cari', 1, 0, 'C');
        $pdf->Cell(85, 7, 'Açıklama', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Tutar', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        
        if (empty($hareketler)) {
            $pdf->Cell(195, 7, 'Kayıt bulunamadı.', 1, 1, 'C');
        } else {
            foreach ($hareketler as $hareket) {
                $pdf->Cell(25, 7, date('d.m.Y', strtotime($hareket['tarih'])), 1, 0, 'C');
                
                $tur_text = '';
                switch ($hareket['tur']) {
                    case 'giris': $tur_text = 'Giriş'; break;
                    case 'cikis': $tur_text = 'Çıkış'; break;
                    case 'transfer': $tur_text = 'Transfer'; break;
                    default: $tur_text = $hareket['tur']; break;
                }
                $pdf->Cell(20, 7, $tur_text, 1, 0, 'C');
                
                $pdf->Cell(40, 7, $hareket['cari_adi'] ? $hareket['cari_adi'] : '-', 1, 0, 'L');
                $pdf->Cell(85, 7, mb_substr($hareket['aciklama'], 0, 45), 1, 0, 'L');
                $pdf->Cell(25, 7, number_format($hareket['tutar'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
            }
        }
        
    } catch (PDOException $e) {
        $pdf->Cell(0, 10, 'Veri çekme hatası: ' . $e->getMessage(), 0, 1, 'C');
    }
}

// Genel Rapor PDF'i oluştur
function generateRapor($pdf, $baslangic_tarihi, $bitis_tarihi, $db) {
    try {
        // PDF başlığı
        $pdf->SetFont('dejavusans', 'B', 16);
        $pdf->Cell(0, 10, 'Finansal Rapor', 0, 1, 'C');
        $pdf->SetFont('dejavusans', '', 10);
        $pdf->Cell(0, 7, date('d.m.Y', strtotime($baslangic_tarihi)) . ' - ' . date('d.m.Y', strtotime($bitis_tarihi)), 0, 1, 'C');
        $pdf->Ln(5);
        
        // Gelir-Gider Özeti
        $stmt_gelir_gider = $db->prepare("
            SELECT 
                SUM(CASE WHEN h.tur = 'giris' THEN h.tutar ELSE 0 END) as toplam_gelir,
                SUM(CASE WHEN h.tur = 'cikis' THEN h.tutar ELSE 0 END) as toplam_gider
            FROM hesap_hareketleri h
            WHERE h.tarih BETWEEN ? AND ?
        ");
        $stmt_gelir_gider->execute([$baslangic_tarihi, $bitis_tarihi]);
        $gelir_gider = $stmt_gelir_gider->fetch(PDO::FETCH_ASSOC);
        
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Gelir-Gider Özeti', 0, 1, 'L');
        $pdf->SetFont('dejavusans', '', 10);
        
        $pdf->Cell(40, 7, 'Toplam Gelir:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($gelir_gider['toplam_gelir'] ?? 0, 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $pdf->Cell(40, 7, 'Toplam Gider:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($gelir_gider['toplam_gider'] ?? 0, 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $net_durum = ($gelir_gider['toplam_gelir'] ?? 0) - ($gelir_gider['toplam_gider'] ?? 0);
        $pdf->Cell(40, 7, 'Net Durum:', 0, 0, 'L');
        $pdf->Cell(0, 7, number_format($net_durum, 2, ',', '.') . ' ' . getParaBirimiSembolu(), 0, 1, 'L');
        
        $pdf->Ln(5);
        
        // Hesap Bakiyeleri
        $hesap_bakiyeleri = $db->query("
            SELECT h.hesap_adi, h.bakiye, h.hesap_turu
            FROM hesaplar h
            ORDER BY h.bakiye DESC
        ")->fetchAll(PDO::FETCH_ASSOC);
        
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Hesap Bakiyeleri', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(80, 7, 'Hesap Adı', 1, 0, 'C');
        $pdf->Cell(40, 7, 'Hesap Türü', 1, 0, 'C');
        $pdf->Cell(30, 7, 'Bakiye', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        $toplam_bakiye = 0;
        
        if (empty($hesap_bakiyeleri)) {
            $pdf->Cell(150, 7, 'Kayıt bulunamadı.', 1, 1, 'C');
        } else {
            foreach ($hesap_bakiyeleri as $hesap) {
                $hesap_turu_text = '';
                switch ($hesap['hesap_turu']) {
                    case 'nakit': $hesap_turu_text = 'Nakit'; break;
                    case 'banka': $hesap_turu_text = 'Banka'; break;
                    case 'kredi_karti': $hesap_turu_text = 'Kredi Kartı'; break;
                    default: $hesap_turu_text = $hesap['hesap_turu']; break;
                }
                
                $pdf->Cell(80, 7, $hesap['hesap_adi'], 1, 0, 'L');
                $pdf->Cell(40, 7, $hesap_turu_text, 1, 0, 'C');
                $pdf->Cell(30, 7, number_format($hesap['bakiye'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
                
                $toplam_bakiye += $hesap['bakiye'];
            }
            
            // Toplam satırı
            $pdf->SetFont('dejavusans', 'B', 9);
            $pdf->Cell(120, 7, 'TOPLAM', 1, 0, 'R');
            $pdf->Cell(30, 7, number_format($toplam_bakiye, 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
        }
        
        $pdf->Ln(5);
        
        // Kategori Bazında Borçlar
        $stmt_kategori = $db->prepare("
            SELECT k.kategori_adi, COALESCE(SUM(b.tutar), 0) as toplam_borc
            FROM kategoriler k
            LEFT JOIN borclar b ON k.id = b.kategori_id AND b.tarih BETWEEN ? AND ?
            GROUP BY k.kategori_adi
            ORDER BY toplam_borc DESC
        ");
        $stmt_kategori->execute([$baslangic_tarihi, $bitis_tarihi]);
        $kategori_borclari = $stmt_kategori->fetchAll(PDO::FETCH_ASSOC);
        
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Kategori Bazında Borçlar', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(120, 7, 'Kategori', 1, 0, 'C');
        $pdf->Cell(30, 7, 'Toplam Borç', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        
        if (empty($kategori_borclari)) {
            $pdf->Cell(150, 7, 'Kayıt bulunamadı.', 1, 1, 'C');
        } else {
            foreach ($kategori_borclari as $kategori) {
                $pdf->Cell(120, 7, $kategori['kategori_adi'], 1, 0, 'L');
                $pdf->Cell(30, 7, number_format($kategori['toplam_borc'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
            }
        }
        
        $pdf->Ln(5);
        
        // Vadesi Geçmiş Borçlar/Alacaklar
        $bugun = date('Y-m-d');
        $stmt_vadesi_gecmis = $db->prepare("
            SELECT 
                b.id, b.tur, b.tutar, b.tarih, b.vade_tarihi, b.aciklama,
                c.ad_soyad as cari_adi
            FROM borclar b
            JOIN cariler c ON b.cari_id = c.id
            WHERE b.vade_tarihi < ? AND b.durum != 'odendi'
            ORDER BY b.vade_tarihi ASC
        ");
        $stmt_vadesi_gecmis->execute([$bugun]);
        $vadesi_gecmis = $stmt_vadesi_gecmis->fetchAll(PDO::FETCH_ASSOC);
        
        $pdf->SetFont('dejavusans', 'B', 12);
        $pdf->Cell(0, 10, 'Vadesi Geçmiş Borçlar/Alacaklar', 0, 1, 'L');
        $pdf->SetFont('dejavusans', 'B', 9);
        
        // Tablo başlıkları
        $pdf->Cell(15, 7, 'Tür', 1, 0, 'C');
        $pdf->Cell(40, 7, 'Cari', 1, 0, 'C');
        $pdf->Cell(25, 7, 'Vade Tarihi', 1, 0, 'C');
        $pdf->Cell(40, 7, 'Açıklama', 1, 0, 'C');
        $pdf->Cell(30, 7, 'Tutar', 1, 1, 'C');
        
        // Tablo içeriği
        $pdf->SetFont('dejavusans', '', 9);
        
        if (empty($vadesi_gecmis)) {
            $pdf->Cell(150, 7, 'Vadesi geçmiş borç veya alacak bulunmamaktadır.', 1, 1, 'C');
        } else {
            foreach ($vadesi_gecmis as $vade) {
                $pdf->Cell(15, 7, $vade['tur'] == 'satis' ? 'Alacak' : 'Borç', 1, 0, 'C');
                $pdf->Cell(40, 7, mb_substr($vade['cari_adi'], 0, 20), 1, 0, 'L');
                $pdf->Cell(25, 7, date('d.m.Y', strtotime($vade['vade_tarihi'])), 1, 0, 'C');
                $pdf->Cell(40, 7, mb_substr($vade['aciklama'], 0, 20), 1, 0, 'L');
                $pdf->Cell(30, 7, number_format($vade['tutar'], 2, ',', '.') . ' ' . getParaBirimiSembolu(), 1, 1, 'R');
            }
        }
        
    } catch (PDOException $e) {
        $pdf->Cell(0, 10, 'Veri çekme hatası: ' . $e->getMessage(), 0, 1, 'C');
    }
}
?>
