<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../functions/log_fonksiyonlari.php';
require_once __DIR__ . '/../functions/yardimci_fonksiyonlar.php';

yetkili_mi('faturalar'); // Yetki kontrolü

global $db;

// 1. ID'yi Al ve Doğrula
if (!isset($_GET['id']) || !($islem_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT))) {
    $_SESSION['hata'] = "Geçersiz İşlem ID.";
    header("Location: index.php");
    exit();
}

$sayfa_baslik = "Stoksuz İşlem Düzenle";
$paraBirimi = getFirmaPara();
$aktif_sirket_id = $_SESSION['aktif_sirket_id'];

// Formdan gelen verileri tutmak için
$form_verileri = [];

try {
    // 2. İşlem Ana Bilgilerini Çek
    $stmt = $db->prepare("SELECT * FROM borclar WHERE id = ? AND sirket_id = ?");
    $stmt->execute([$islem_id, $aktif_sirket_id]);
    $islem = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$islem) {
        $_SESSION['hata'] = "İşlem bulunamadı veya bu işlemi düzenleme yetkiniz yok.";
        header("Location: index.php");
        exit();
    }

    // Bu sayfa sadece stoksuz işlemler içindir.
    if ($islem['stoklu_islem'] == 1) {
        header("Location: borc_duzenle.php?id=" . $islem_id);
        exit();
    }

} catch (PDOException $e) {
    error_log("İşlem düzenleme sayfası (veri çekme) hatası: " . $e->getMessage());
    $_SESSION['hata'] = "Veritabanı hatası: Veriler yüklenemedi.";
    header("Location: index.php");
    exit();
}

// Form Gönderildiğinde (POST Metodu)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
        header("Location: islem_duzenle.php?id=" . $islem_id);
        exit();
    }

    // Form verilerini al ve işle
    $form_verileri = $_POST;
    $hatalar = [];

    $cari_id = filter_input(INPUT_POST, 'cari_id', FILTER_VALIDATE_INT);
    $kategori_id = filter_input(INPUT_POST, 'kategori_id', FILTER_VALIDATE_INT);
    $tarih = filter_input(INPUT_POST, 'tarih', FILTER_SANITIZE_SPECIAL_CHARS);
    $vade_tarihi = filter_input(INPUT_POST, 'vade_tarihi', FILTER_SANITIZE_SPECIAL_CHARS) ?: null;
    
    // Dinamik kalemleri işle
    $aciklamalar = $_POST['aciklama'] ?? [];
    $tutarlar = $_POST['tutar'] ?? [];
    $toplam_tutar = 0;
    $birlesik_aciklama = [];

    foreach ($tutarlar as $i => $tutar_str) {
        $tutar = filter_var(str_replace(',', '.', $tutar_str), FILTER_VALIDATE_FLOAT);
        if ($tutar > 0) {
            $toplam_tutar += $tutar;
            $aciklama = temizle($aciklamalar[$i]);
            if (!empty($aciklama)) {
                $birlesik_aciklama[] = $aciklama . ' (' . number_format($tutar, 2, ',', '.') . ' ' . $paraBirimi . ')';
            }
        } elseif (!empty($aciklamalar[$i])) {
            $birlesik_aciklama[] = $aciklamalar[$i];
        }
    }
    $genel_aciklama = implode("\n", $birlesik_aciklama);

    // Doğrulama
    if (empty($cari_id)) $hatalar[] = "Cari seçimi zorunludur.";
    if (empty($kategori_id)) $hatalar[] = "Kategori seçimi zorunludur.";
    if (empty($tarih)) $hatalar[] = "İşlem tarihi zorunludur.";
    if ($toplam_tutar <= 0) $hatalar[] = "Genel toplam sıfırdan büyük olmalıdır.";

    if (empty($hatalar)) {
        try {
            $db->beginTransaction();

            // 1. Eski cari bakiyesini ters çevir
            $eski_tutar = $islem['tutar'];
            $borc_degisim_eski = ($islem['tur'] == 'satis' || $islem['tur'] == 'gider') ? -$eski_tutar : 0;
            $alacak_degisim_eski = ($islem['tur'] == 'alis' || $islem['tur'] == 'gelir') ? -$eski_tutar : 0;
            cariBakiyeGuncelle($db, $islem['cari_id'], $borc_degisim_eski, $alacak_degisim_eski);

            // 2. Ana tabloyu (borclar) güncelle
            $stmt_update = $db->prepare("UPDATE borclar SET cari_id=?, kategori_id=?, tutar=?, aciklama=?, tarih=?, vade_tarihi=? WHERE id=?");
            $stmt_update->execute([$cari_id, $kategori_id, $toplam_tutar, $genel_aciklama, $tarih, $vade_tarihi, $islem_id]);

            // 3. Yeni cari bakiyesini uygula
            $borc_degisim_yeni = ($islem['tur'] == 'satis' || $islem['tur'] == 'gider') ? $toplam_tutar : 0;
            $alacak_degisim_yeni = ($islem['tur'] == 'alis' || $islem['tur'] == 'gelir') ? $toplam_tutar : 0;
            cariBakiyeGuncelle($db, $cari_id, $borc_degisim_yeni, $alacak_degisim_yeni);

            $db->commit();
            sistem_log_ekle($db, 'update', 'borclar', $islem_id, json_encode($islem), json_encode($_POST), "{$islem_id} ID'li stoksuz işlem güncellendi.");
            $_SESSION['mesaj'] = "İşlem başarıyla güncellendi.";
            header("Location: borc_detay.php?id=" . $islem_id);
            exit();

        } catch (Exception $e) {
            $db->rollBack();
            error_log("Stoksuz işlem güncelleme hatası: " . $e->getMessage());
            $hatalar[] = "Güncelleme sırasında bir veritabanı hatası oluştu: " . $e->getMessage();
        }
    }
    $_SESSION['form_hatalari'] = $hatalar;

} else {
    $form_verileri = $islem;
    // Sayfa ilk yüklendiğinde kalemleri hazırlayalım
    $aciklama_db = $islem['aciklama'];
    $aciklama_normalized = str_replace('\n', "\n", $aciklama_db);

    // 1. Önce \n ile böl
    $aciklama_satirlari = explode("\n", $aciklama_normalized);

    // 2. Eğer tek satır varsa ve içinde birden fazla 'Borç bilgisi' varsa, regex ile böl
    if (count($aciklama_satirlari) === 1 && preg_match_all('/(Borç bilgisi [^\\(]+\\([^)]+\\))/u', $aciklama_normalized, $matches)) {
        $aciklama_satirlari = $matches[1];
    }
    $form_verileri['kalemler'] = [];
    if (!empty($aciklama_satirlari)) {
        foreach ($aciklama_satirlari as $satir) {
            // Açıklama ve tutarı regex ile ayır
            if (preg_match('/^(.*)\\(([^)]+)\\)$/u', $satir, $parcalar)) {
                $aciklama = trim($parcalar[1]);
                $tutar = trim(str_replace(['USD','TL','TRY','EUR'], '', $parcalar[2]));
                $tutar = str_replace([','], ['.'], $tutar); // Nokta/virgül düzeltmesi
                $form_verileri['kalemler'][] = ['aciklama' => $aciklama, 'tutar' => $tutar];
            } else {
                // Tutar yoksa açıklamayı yaz, tutarı 0 ata
                $form_verileri['kalemler'][] = ['aciklama' => $satir, 'tutar' => '0'];
            }
        }
    } else {
        // Açıklama yoksa bile boş bir satır göster
        $form_verileri['kalemler'][] = ['aciklama' => '', 'tutar' => $islem['tutar']];
    }
}

// Form için gerekli diğer verileri çek
$cariler = $db->query("SELECT id, ad_soyad FROM cariler WHERE sirket_id = {$aktif_sirket_id} ORDER BY ad_soyad")->fetchAll(PDO::FETCH_ASSOC);
$kategoriler = $db->query("SELECT id, kategori_adi FROM kategoriler WHERE sirket_id = {$aktif_sirket_id} ORDER BY kategori_adi")->fetchAll(PDO::FETCH_ASSOC);

$csrf_token = csrf_token_olustur();
$form_hatalari = $_SESSION['form_hatalari'] ?? [];
unset($_SESSION['form_hatalari']);

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

<div class="container mt-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h2><i class="bi bi-pencil-square me-2"></i><?= htmlspecialchars($sayfa_baslik) ?> (#<?= $islem_id ?>)</h2>
        <a href="borc_detay.php?id=<?= $islem_id ?>" class="btn btn-secondary"><i class="bi bi-arrow-left"></i> İşlem Detayına Dön</a>
    </div>

    <?php if (!empty($form_hatalari)): ?>
        <div class="alert alert-danger">
            <strong>Lütfen hataları düzeltin:</strong>
            <ul><?php foreach ($form_hatalari as $hata): ?><li><?= htmlspecialchars($hata) ?></li><?php endforeach; ?></ul>
        </div>
    <?php endif; ?>

    <form method="POST" id="islemForm">
        <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">

        <div class="card shadow-sm mb-4">
            <div class="card-header"><h5 class="mb-0">Temel Bilgiler</h5></div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="cari_id" class="form-label">Cari (Müşteri/Tedarikçi) <span class="text-danger">*</span></label>
                        <select class="form-select" id="cari_id" name="cari_id" required>
                            <?php foreach ($cariler as $cari): ?>
                                <option value="<?= $cari['id'] ?>" <?= ($form_verileri['cari_id'] ?? '') == $cari['id'] ? 'selected' : '' ?>><?= htmlspecialchars($cari['ad_soyad']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label for="kategori_id" class="form-label">Kategori <span class="text-danger">*</span></label>
                        <select class="form-select" id="kategori_id" name="kategori_id" required>
                            <?php foreach ($kategoriler as $kategori): ?>
                                <option value="<?= $kategori['id'] ?>" <?= ($form_verileri['kategori_id'] ?? '') == $kategori['id'] ? 'selected' : '' ?>><?= htmlspecialchars($kategori['kategori_adi']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label for="tarih" class="form-label">İşlem Tarihi <span class="text-danger">*</span></label>
                        <input type="date" class="form-control" id="tarih" name="tarih" value="<?= htmlspecialchars($form_verileri['tarih'] ?? date('Y-m-d')) ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label for="vade_tarihi" class="form-label">Vade Tarihi</label>
                        <input type="date" class="form-control" id="vade_tarihi" name="vade_tarihi" value="<?= htmlspecialchars($form_verileri['vade_tarihi'] ?? '') ?>">
                    </div>
                </div>
            </div>
        </div>

        <div class="card shadow-sm mb-4">
            <div class="card-header bg-light d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><i class="bi bi-journal-text me-2"></i>İşlem Kalemleri</h5>
                <button type="button" class="btn btn-sm btn-outline-primary" id="addKalemRow"><i class="bi bi-plus-circle me-1"></i> Kalem Ekle</button>
            </div>
            <div class="card-body">
                <div id="kalemRows">
                    <?php 
                    $kalemler = $form_verileri['kalemler'] ?? [['aciklama' => '', 'tutar' => '']];
                    foreach ($kalemler as $kalem): 
                    ?>
                    <div class="row kalem-row mb-2">
                        <div class="col-md-7">
                            <label class="form-label">Açıklama</label>
                            <input type="text" class="form-control" name="aciklama[]" placeholder="Örn: Danışmanlık hizmeti" value="<?= htmlspecialchars($kalem['aciklama']) ?>">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Tutar (<?= $paraBirimi ?>)</label>
                            <input type="text" class="form-control tutar-input" name="tutar[]" placeholder="0,00" value="<?= number_format((float)$kalem['tutar'], 2, ',', '.') ?>" required>
                        </div>
                        <div class="col-md-1 d-flex align-items-end">
                            <button type="button" class="btn btn-outline-danger remove-kalem-row"><i class="bi bi-x-circle"></i></button>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <hr>
                <div class="row justify-content-end">
                    <div class="col-md-5">
                        <div class="toplam-alan p-3">
                            <ul class="list-group">
                                <li class="list-group-item d-flex justify-content-between align-items-center bg-light">
                                    <strong>Genel Toplam:</strong> <strong id="genelToplam">0,00 <?= $paraBirimi ?></strong>
                                </li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="d-flex justify-content-end mb-4">
            <button type="submit" class="btn btn-primary btn-lg"><i class="bi bi-save me-2"></i>İşlemi Güncelle</button>
        </div>
    </form>
</div>

<?php include __DIR__ . "/../includes/footer.php"; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/jquery@3.7.1/dist/jquery.min.js"></script>
<script>
$(document).ready(function() {
    const paraBirimi = '<?= $paraBirimi ?>';

    function formatCurrency(value) {
        return new Intl.NumberFormat('tr-TR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(value);
    }

    function parseFloatOrZero(value) {
        if (typeof value !== 'string' || value.trim() === '') return 0;
        const cleanedValue = value.replace(/\./g, '').replace(',', '.');
        const parsed = parseFloat(cleanedValue);
        return isNaN(parsed) ? 0 : parsed;
    }

    function updateGenelToplam() {
        let genelToplam = 0;
        $('.tutar-input').each(function() {
            genelToplam += parseFloatOrZero($(this).val());
        });
        $('#genelToplam').text(formatCurrency(genelToplam) + ' ' + paraBirimi);
    }

    $('#addKalemRow').on('click', function() {
        const newRow = `
        <div class="row kalem-row mb-2">
            <div class="col-md-7">
                <label class="form-label">Açıklama</label>
                <input type="text" class="form-control" name="aciklama[]" placeholder="Örn: Diğer masraflar">
            </div>
            <div class="col-md-4">
                <label class="form-label">Tutar (${paraBirimi})</label>
                <input type="text" class="form-control tutar-input" name="tutar[]" placeholder="0,00" required>
            </div>
            <div class="col-md-1 d-flex align-items-end">
                <button type="button" class="btn btn-outline-danger remove-kalem-row"><i class="bi bi-x-circle"></i></button>
            </div>
        </div>`;
        $('#kalemRows').append(newRow);
    });

    $('#kalemRows').on('click', '.remove-kalem-row', function() {
        if ($('.kalem-row').length > 1) {
            $(this).closest('.kalem-row').remove();
            updateGenelToplam();
        } else {
            alert('En az bir kalem kalmalıdır.');
        }
    });

    $('#kalemRows').on('input', '.tutar-input', function() {
        updateGenelToplam();
    });

    // Initial calculation
    updateGenelToplam();
});
</script>
</body>
</html>
