<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi('faturalar'); // Invoice view permission

// tFPDF kütüphanesini dahil et
require_once __DIR__ . '/tfpdf/tfpdf.php';

// --- Veritabanı İşlemleri Başlangıç ---

$fatura_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$fatura_id) {
    die(__('error_invalid_invoice_id'));
}

$aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;
$fatura = null;
$detaylar = [];
$sirket_bilgileri = null;
$cari = null;

try {
    // Şirket Bilgileri
    $stmt_sirket = $db->prepare("SELECT * FROM sirketler WHERE id = ?");
    $stmt_sirket->execute([$aktif_sirket_id]);
    $sirket_bilgileri = $stmt_sirket->fetch(PDO::FETCH_ASSOC);

    // Fatura (borç) ana bilgileri
    $stmt_fatura = $db->prepare("SELECT * FROM borclar WHERE id = ? AND sirket_id = ?");
    $stmt_fatura->execute([$fatura_id, $aktif_sirket_id]);
    $fatura = $stmt_fatura->fetch(PDO::FETCH_ASSOC);

    if (!$fatura) {
        die(__('error_invoice_not_found'));
    }

    // Cari bilgileri
    $stmt_cari = $db->prepare("SELECT * FROM cariler WHERE id = ? AND sirket_id = ?");
    $stmt_cari->execute([$fatura['cari_id'], $aktif_sirket_id]);
    $cari = $stmt_cari->fetch(PDO::FETCH_ASSOC);
    // Fatura detayları (ürünler) - kur bilgisi de dahil
    $stmt_detaylar = $db->prepare("
        SELECT fd.*, u.urun_adi, u.birim, 
               CASE 
                   WHEN b.tur = 'satis' THEN u.satis_para_birimi 
                   ELSE u.alis_para_birimi 
               END as urun_para_birimi
        FROM fatura_detaylari fd 
        JOIN urunler u ON fd.urun_id = u.id 
        JOIN borclar b ON fd.borc_id = b.id
        WHERE fd.borc_id = ?
    ");
    $stmt_detaylar->execute([$fatura_id]);
    $detaylar = $stmt_detaylar->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    die(__('database_error') . ": " . $e->getMessage());
}

// --- Veritabanı İşlemleri Bitiş ---

// --- PDF Sınıfı ve Oluşturma Başlangıç ---
class PDF_Invoice extends tFPDF
{
    public $sirket_bilgileri;
    private $widths;
    private $aligns;

    function SetWidths($w) {
        $this->widths = $w;
    }

    function SetAligns($a) {
        $this->aligns = $a;
    }

    function Row($data, $fill = false, $border = 1, $height = 7) {
        $nb = 0;
        
        // Satır için gerekli yüksekliği hesapla
        for($i = 0; $i < count($data); $i++) {
            $nb = max($nb, $this->NbLines($this->widths[$i], $data[$i]));
        }
        $h = $height * $nb;
        
        // Gerekirse sayfa sonu ekle
        $this->CheckPageBreak($h);
        
        // Hücrelerin sınır çizgilerini çiz
        $x = $this->GetX();
        $y = $this->GetY();
        
        // Tüm hücrelerin dış çerçevesini çiz
        $totalWidth = array_sum($this->widths);
        $this->Rect($x, $y, $totalWidth, $h, 'DF', array('all' => array('width' => 0.2)));
        
        // Hücre içi çizgileri çiz
        $currentX = $x;
        for($i = 0; $i < count($this->widths) - 1; $i++) {
            $currentX += $this->widths[$i];
            $this->Line($currentX, $y, $currentX, $y + $h);
        }
        
        // Hücre içeriğini yazdır
        for($i = 0; $i < count($data); $i++) {
            $w = $this->widths[$i];
            $a = $this->aligns[$i] ?? 'L';
            
            // Metin rengini ayarla
            $this->SetTextColor(0, 0, 0);
            
            // Hücre içeriğini yazdır
            $this->SetXY($x, $y);
            $this->MultiCell($w, $height, $data[$i], 0, $a, $fill);
            
            $x += $w;
        }
        
        // Bir sonraki satıra geç
        $this->SetXY($this->lMargin, $y + $h);
    }
    
    function CheckPageBreak($h) {
        if($this->GetY() + $h > $this->PageBreakTrigger) {
            $this->AddPage($this->CurOrientation);
            return true;
        }
        return false;
    }
    
    // UTF-8 karakterler için ord() işlevi
    function utf8Ord($u) {
        $k = mb_convert_encoding($u, 'UCS-2LE', 'UTF-8');
        $k1 = ord(substr($k, 0, 1));
        $k2 = ord(substr($k, 1, 1));
        return $k2 * 256 + $k1;
    }
    
    function NbLines($w, $txt) {
        // Eğer metin sayısal bir değerse string'e çevir
        if (is_numeric($txt) || is_bool($txt)) {
            $txt = (string)$txt;
        }
        
        // Eğer hala string değilse veya boşsa, 1 döndür
        if (!is_string($txt) || $txt === '') {
            return 1;
        }
        
        // Eğer genişlik 0'dan küçükse, 1 döndür
        if ($w <= 0) {
            return 1;
        }
        
        // Font bilgilerini al
        if (!isset($this->CurrentFont) || !isset($this->CurrentFont['cw'])) {
            return 1;
        }
        
        $cw = $this->CurrentFont['cw'];
        
        // Maksimum genişliği hesapla
        $wmax = ($w - 2 * $this->cMargin) * 1000 / $this->FontSize;
        
        // Metni temizle ve satır sonlarını düzenle
        $s = str_replace("\r", '', $txt);
        
        // Eğer metin boşsa 1 döndür
        if ($s === '') {
            return 1;
        }
        
        // Eğer son karakter yeni satır ise kaldır
        $s = rtrim($s, "\n");
        
        // Karakterlere ayır
        $chars = [];
        $len = mb_strlen($s, 'UTF-8');
        for ($i = 0; $i < $len; $i++) {
            $chars[] = mb_substr($s, $i, 1, 'UTF-8');
        }
        
        $nb = count($chars);
        if ($nb === 0) {
            return 1;
        }
        
        $sep = -1;
        $i = 0;
        $j = 0;
        $l = 0;
        $nl = 1;
        
        while ($i < $nb) {
            $c = $chars[$i];
            
            if ($c === "\n") {
                $i++;
                $sep = -1;
                $j = $i;
                $l = 0;
                $nl++;
                continue;
            }
            
            if ($c === ' ') {
                $sep = $i;
            }
            
            // Karakter genişliğini güvenli bir şekilde al
            $charWidth = 0;
            $ord = $this->utf8Ord($c);
            
            if ($ord !== false && isset($cw[$ord])) {
                $charWidth = $cw[$ord];
            } else {
                // Eğer karakter genişliği bilinmiyorsa, varsayılan bir değer kullan
                $charWidth = 600; // Varsayılan genişlik
            }
            
            $l = (float)$l + (float)$charWidth;
            
            if ($l > $wmax) {
                if ($sep == -1) {
                    if ($i == $j) {
                        $i++;
                    }
                } else {
                    $i = $sep + 1;
                }
                $sep = -1;
                $j = $i;
                $l = 0;
                $nl++;
            } else {
                $i++;
            }
        }
        return max(1, $nl); // En az 1 satır döndür
    }

    function Header()
    {
        // Logo sol tarafta
        if (!empty($this->sirket_bilgileri['logo_yolu']) && file_exists(__DIR__ . '/../' . $this->sirket_bilgileri['logo_yolu'])) {
            $this->Image(__DIR__ . '/../' . $this->sirket_bilgileri['logo_yolu'], 15, 10, 40);
        }
        
        // Fatura başlığını sağ üst köşeye yerleştir
        $this->SetY(15); // Logo ile aynı hizaya getir
        $this->SetFont('DejaVu', 'B', 16);
        $this->SetX(-60); // Sağ kenardan 60mm içeri
        $this->Cell(0, 10, strtoupper(__('invoice')), 0, 0, 'R');
        $this->Ln(25); // Alt satıra geç ve biraz boşluk bırak
    }

    function Footer()
    {
        if (!empty($this->sirket_bilgileri)) {
            $this->SetY(-20);
            $this->SetFont('DejaVu', '', 8);
            $this->SetTextColor(100, 100, 100);
            
            // Safely get company info with fallback to empty string if not set
            $firmaAdi = $this->sirket_bilgileri['firma_adi'] ?? '';
            $adres = $this->sirket_bilgileri['adres'] ?? '';
            $telefon = $this->sirket_bilgileri['telefon'] ?? '';
            $email = $this->sirket_bilgileri['email'] ?? '';
            
            // Only show company name and address if we have at least one of them
            if (!empty($firmaAdi) || !empty($adres)) {
                $this->Cell(0, 5, trim($firmaAdi . (!empty($firmaAdi) && !empty($adres) ? ' - ' : '') . $adres), 0, 1, 'C');
            }
            
            // Only show phone and email if we have at least one of them
            if (!empty($telefon) || !empty($email)) {
                $phoneText = !empty($telefon) ? 'Tel: ' . $telefon : '';
                $emailText = !empty($email) ? 'E-posta: ' . $email : '';
                $separator = (!empty($telefon) && !empty($email)) ? ' | ' : '';
                $this->Cell(0, 5, $phoneText . $separator . $emailText, 0, 1, 'C');
            }
            
            // Always show page number
            $this->Cell(0, 5, 'Sayfa ' . $this->PageNo() . '/{nb}', 0, 0, 'C');
        }
    }
}

$pdf = new PDF_Invoice();
$pdf->sirket_bilgileri = $sirket_bilgileri;
$pdf->AddFont('DejaVu', '', 'DejaVuSans.ttf', true);
$pdf->AddFont('DejaVu', 'B', 'DejaVuSans-Bold.ttf', true);
$pdf->AliasNbPages();
$pdf->AddPage();

// Satıcı ve Alıcı Bilgileri
$satici_etiket = $fatura['tur'] == 'satis' ? strtoupper(__('seller_info')) : strtoupper(__('buyer_info'));
$alici_etiket = $fatura['tur'] == 'satis' ? strtoupper(__('buyer_info')) : strtoupper(__('seller_info'));

$satici = $fatura['tur'] == 'satis' ? $sirket_bilgileri : $cari;
$alici = $fatura['tur'] == 'satis' ? $cari : $sirket_bilgileri;

$pdf->SetFont('DejaVu', 'B', 10);
$pdf->Cell(95, 7, $satici_etiket, 0, 0);
$pdf->Cell(95, 7, $alici_etiket, 0, 1);

$pdf->SetFont('DejaVu', '', 10);
$pdf->Cell(95, 6, $satici['sirket_adi'] ?? $satici['ad_soyad'], 0, 0);
$pdf->Cell(95, 6, $alici['ad_soyad'] ?? $alici['sirket_adi'], 0, 1);

// Adresleri ve diğer bilgileri dinamik olarak hizala
$y_baslangic = $pdf->GetY();
$pdf->MultiCell(95, 5, $satici['adres'] ?? '', 0, 'L');
$pdf->Cell(95, 5, $satici['telefon'] ?? '', 0, 1);
$pdf->Cell(95, 5, $satici['email'] ?? '', 0, 1);
$y_bitis_satici = $pdf->GetY();

$pdf->SetXY(105, $y_baslangic);

$pdf->MultiCell(95, 5, $alici['adres'] ?? '', 0, 'L');
$pdf->SetX(105);
$pdf->Cell(95, 5, $alici['telefon'] ?? '', 0, 1);
$pdf->SetX(105);
$pdf->Cell(95, 5, $alici['email'] ?? '', 0, 1);
$y_bitis_alici = $pdf->GetY();

// Bir sonraki eleman için en uzun sütunun bittiği Y pozisyonunu ayarla
$pdf->SetY(max($y_bitis_satici, $y_bitis_alici));
$pdf->Ln(10);

// Fatura Detayları
$pdf->SetFont('DejaVu', 'B', 10);
$pdf->Cell(40, 7, __('invoice_number') . ':', 1, 0);
$pdf->SetFont('DejaVu', '', 10);
$pdf->Cell(55, 7, '#' . str_pad($fatura['id'], 6, '0', STR_PAD_LEFT), 1, 0);
$pdf->SetFont('DejaVu', 'B', 10);
$pdf->Cell(40, 7, __('invoice_date') . ':', 1, 0);
$pdf->SetFont('DejaVu', '', 10);
$pdf->Cell(55, 7, date('d.m.Y', strtotime($fatura['tarih'])), 1, 1);

// Ürün/Hizmet Tablosu
if ($fatura['stoklu_islem'] == '1' && !empty($detaylar)) {
    $pdf->Ln(10);
    $pdf->SetFont('DejaVu', 'B', 10);
    $pdf->SetFont('DejaVu', 'B', 9);
    $pdf->SetFillColor(79, 129, 189); // Mavi arka plan
    $pdf->SetTextColor(255, 255, 255); // Beyaz yazı
    $pdf->SetDrawColor(79, 129, 189); // Kenarlık rengi

    // Tablo genişliklerini ve hizalamalarını ayarla
    $widths = array(10, 70, 20, 30, 25, 35);
    $aligns = array('C', 'L', 'C', 'R', 'R', 'R');
    $pdf->SetWidths($widths);
    $pdf->SetAligns($aligns);

    // Başlık satırı için stil ayarları
    $pdf->SetFillColor(79, 129, 189); // Mavi arka plan
    $pdf->SetTextColor(255, 255, 255); // Beyaz yazı
    $pdf->SetLineWidth(0.2); // İnce çizgi kalınlığı
    $pdf->SetDrawColor(100, 100, 100); // Koyu gri çizgi rengi

    // Başlık satırı
    $pdf->Row(
        array(
            '#',
            __('product_name'),
            __('quantity'),
            __('unit_price'),
            __('tax'),
            __('amount')
        ),
        true // Dolgu rengini uygula
    );

    // Veri satırları için stil ayarları
    $pdf->SetFillColor(255, 255, 255); // Beyaz arka plan
    $pdf->SetTextColor(0, 0, 0); // Siyah yazı
    $pdf->SetDrawColor(200, 200, 200); // Açık gri çizgi rengi

    // Ürün satırlarını oluştururken kullanılan döviz kurlarını topla
    $kullanilanKurlar = [];
    $paraBirimi = $fatura['para_birimi'] ?? 'TL';
    
    foreach ($detaylar as $detay) {
        $urunParaBirimi = $detay['urun_para_birimi'] ?? 'TL';
        $urunParaBirimi = $urunParaBirimi === 'TRY' ? 'TL' : $urunParaBirimi;
        
        if ($urunParaBirimi !== $paraBirimi && !in_array($urunParaBirimi, array_column($kullanilanKurlar, 'para_birimi'))) {
            $stmt_kur = $db->prepare("
                SELECT alis, satis FROM doviz_kurlari 
                WHERE para_birimi = ? 
                ORDER BY guncelleme_tarihi DESC 
                LIMIT 1
            
            ");
            $stmt_kur->execute([$detay['urun_para_birimi']]);
            $doviz_kuru = $stmt_kur->fetch(PDO::FETCH_ASSOC);
            
            if ($doviz_kuru) {
                $kur = ($fatura['tur'] == 'satis') ? (float)$doviz_kuru['satis'] : (float)$doviz_kuru['alis'];
                $kullanilanKurlar[] = [
                    'para_birimi' => $urunParaBirimi,
                    'kur' => $kur
                ];
            }
        }
    }

    // Ürün satırları
    $pdf->SetFont('DejaVu', '', 9);
    $fill = false; // Satır renklendirmesi için
    $rowCount = 0; // Satır sayacı
    
    // Fatura ayarlarını kontrol et (ürün adetlerini göster ayarı)
    $show_quantities = true; // Varsayılan olarak göster
    try {
        $stmt_ayar = $db->prepare("SELECT ayar_degeri FROM ayarlar WHERE ayar_adi = 'fatura_urun_adetleri_goster' AND (sirket_id = ? OR sirket_id IS NULL) ORDER BY sirket_id DESC LIMIT 1");        
        $stmt_ayar->execute([$aktif_sirket_id]);
        $ayar = $stmt_ayar->fetch(PDO::FETCH_ASSOC);
        if ($ayar && $ayar['ayar_degeri'] === '0') {
            $show_quantities = false;
        }
    } catch (Exception $e) {
        // Hata durumunda varsayılan değeri kullan
        error_log("Fatura ayarları yüklenirken hata: " . $e->getMessage());
    }

    foreach ($detaylar as $index => $detay) {
        $urunParaBirimi = $detay['urun_para_birimi'] ?? 'TL';
        $urunParaBirimi = $urunParaBirimi === 'TRY' ? 'TL' : $urunParaBirimi;
        $birimFiyat = $detay['birim_fiyat'];
        $tutar = $detay['miktar'] * $birimFiyat;
        $ceviriNotu = '';
        
        // Döviz kuru dönüşümü
        if ($urunParaBirimi !== $paraBirimi) {
            // Kullanılan kurlar arasından ilgili döviz kurunu bul
            $kur = null;
            foreach ($kullanilanKurlar as $kurBilgisi) {
                if ($kurBilgisi['para_birimi'] === $urunParaBirimi) {
                    $kur = $kurBilgisi['kur'];
                    break;
                }
            }
            
            if ($kur !== null) {
                $cevrilmisFiyat = $birimFiyat * $kur;
                $cevrilmisTutar = $tutar * $kur;
            }
        }

        // Sadece ürün adını al, döviz çeviri notunu kaldır
        // Ürün adını ve miktarını ayarla
        $urunAdi = $detay['urun_adi'];
        
        // Miktarı formatla
        $miktarDegeri = rtrim(rtrim(number_format($detay['miktar'], 2, ',', '.'), '0'), ',');
        $miktar = $miktarDegeri . ' ' . $detay['birim'];
        
        // Removed redundant quantity display next to product name
        // as it's already shown in the quantity column
        
        // Birim fiyatı formatla
        $birimFiyatGoster = number_format($birimFiyat, 2, ',', '.') . ' ' . $urunParaBirimi;
        
        // KDV tutarını formatla
        $kdvGoster = number_format($detay['kdv_tutari'], 2, ',', '.') . ' ' . $paraBirimi;
        
        // Toplam tutarı hesapla ve formatla
        $toplamTutar = isset($cevrilmisTutar) ? $cevrilmisTutar + $detay['kdv_tutari'] : $tutar + $detay['kdv_tutari'];
        $toplamGoster = number_format($toplamTutar, 2, ',', '.') . ' ' . $paraBirimi;
        
        // Ürün satırını yazdır
        $pdf->Row(
            array(
                $index + 1,
                $urunAdi,
                $miktar,
                $birimFiyatGoster,
                $kdvGoster,
                $toplamGoster
            ),
            $fill,
            1, // Kenarlık kalınlığı
            7   // Satır yüksekliği
        );
        $fill = !$fill; // Bir sonraki satır için dolgu rengini değiştir
        $rowCount++;
        $fill = !$fill;
        
    }
    
    // Toplam ürün adedini hesapla (gösterim için sonraya ertelendi)
    if ($show_quantities) {
        $toplamUrunAdedi = array_sum(array_column($detaylar, 'miktar'));
    }
} else { // Stoksuz işlem ise açıklama kalemlerini tablo olarak göster
    $pdf->Ln(10);
    $pdf->SetFont('DejaVu', 'B', 10);
    $pdf->SetFillColor(33, 150, 243); // Mavi başlık
    $pdf->SetTextColor(255, 255, 255); // Beyaz yazı
    $pdf->Cell(10, 7, '#', 1, 0, 'C', true);
    $pdf->Cell(140, 7, __('description'), 1, 0, 'C', true);
    $pdf->Cell(40, 7, __('amount'), 1, 1, 'C', true);
    $pdf->SetFont('DejaVu', '', 9);
    $pdf->SetTextColor(0,0,0);
    $satirlar = preg_split('/\r\n|\r|\n/', $fatura['aciklama']);
    $num = 1; $bos = true;
    foreach ($satirlar as $satir) {
        $satir = trim($satir);
        if ($satir === '') continue;
        $aciklama = $satir;
        $tutar = '';
        if (preg_match('/^(.*)\(([^)]+)\)$/u', $satir, $parcalar)) {
            $aciklama = trim($parcalar[1]);
            $tutar = trim($parcalar[2]);
        }
        $pdf->Cell(10, 7, $num++, 1, 0, 'C');
        $pdf->Cell(140, 7, $aciklama, 1, 0, 'L');
        $pdf->Cell(40, 7, $tutar, 1, 1, 'R');
        $bos = false;
    }
    if ($bos) {
        $pdf->Cell(190, 7, __('no_description_items'), 1, 1, 'C');
    }
}
$pdf->Ln(10);

// Toplamlar bölümü için boşluk bırak
$pdf->Ln(5);

// Döviz kurlarını göster
if (!empty($kullanilanKurlar)) {
    $pdf->Ln(5);
    $pdf->SetFont('DejaVu', '', 8);
    $pdf->Cell(0, 5, __('exchange_rates') . ':', 0, 1);
    
    foreach ($kullanilanKurlar as $kurBilgisi) {
        $pdf->Cell(0, 4, "1 {$kurBilgisi['para_birimi']} = " . number_format($kurBilgisi['kur'], 4, ',', '.') . " $paraBirimi", 0, 1);
    }
    
    // Toplam ürün adedini döviz kurlarından sonra göster
    if ($show_quantities && isset($toplamUrunAdedi)) {
        $pdf->Ln(3);
        $pdf->Cell(0, 5, __('total_products') . ': ' . number_format($toplamUrunAdedi, 0, ',', '.') . ' ' . mb_strtolower(__('items')), 0, 1);
    }
    
    $pdf->Ln(3);
}

// Toplamlar
$pdf->SetY(-60 - (count($kullanilanKurlar) * 5)); // Döviz kurlarına göre pozisyonu ayarla
$pdf->SetFont('DejaVu', '', 10);
$pdf->SetFillColor(240, 240, 240); // Açık gri arka plan

// Ara toplam
$pdf->Cell(130, 8, __('subtotal') . ':', 1, 0, 'R', true);
$pdf->Cell(60, 8, number_format($fatura['tutar'], 2, ',', '.') . ' ' . $paraBirimi, 1, 1, 'R', true);

// KDV toplamı
if ($fatura['toplam_kdv'] > 0) {
    $pdf->Cell(130, 8, __('total_tax') . ':', 1, 0, 'R', true);
    $pdf->Cell(60, 8, number_format($fatura['toplam_kdv'], 2, ',', '.') . ' ' . $paraBirimi, 1, 1, 'R', true);
}

// İndirim
if ($fatura['indirim_tutari'] > 0) {
    $pdf->Cell(130, 8, __('discount') . ':', 1, 0, 'R', true);
    $pdf->Cell(60, 8, '-' . number_format($fatura['indirim_tutari'], 2, ',', '.') . ' ' . $paraBirimi, 1, 1, 'R', true);
}

$genel_toplam = $fatura['tutar'] + $fatura['toplam_kdv'] - $fatura['indirim_tutari'];

// Çizgi çek
$pdf->SetDrawColor(100, 100, 100);
$pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
$pdf->Ln(2);

// Genel Toplam
$pdf->SetFont('DejaVu', 'B', 12);
$pdf->Cell(130, 12, strtoupper(__('total_amount')) . ':', 0, 0, 'R');
$pdf->SetFillColor(79, 129, 189); // Mavi arka plan
$pdf->SetTextColor(255, 255, 255); // Beyaz yazı
$pdf->Cell(60, 12, number_format($genel_toplam, 2, ',', '.') . ' ' . $paraBirimi, 1, 1, 'C', true);

// Yazı rengini sıfırla
$pdf->SetTextColor(0, 0, 0);

$pdf->Output('I', 'fatura-' . $fatura_id . '.pdf');

// --- PDF Sınıfı ve Oluşturma Bitiş ---
?>
