<?php
/**
 * MIGRATION SCRIPT: Decimal → Integer (Kuruş)
 * 
 * Bu script tüm para değerlerini decimal'den integer'a çevirir
 * 
 * UYARI: Mutlaka yedek alın!
 * 
 * @package Odjoo
 * @version 2.0.0
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/classes/MoneyHelper.php';
require_once __DIR__ . '/classes/TaxEngine.php';

set_time_limit(0);
ini_set('memory_limit', '512M');

// Sadece CLI'dan çalıştırılabilir (güvenlik)
if (php_sapi_name() !== 'cli' && !isset($_GET['force'])) {
    die("Bu script sadece komut satırından çalıştırılabilir. Web'den çalıştırmak için ?force=1 parametresi ekleyin.");
}

echo "==========================================================\n";
echo "  ODJOO - DECIMAL TO INTEGER MIGRATION\n";
echo "  Production-Grade KDV Sistemi\n";
echo "==========================================================\n\n";

echo "UYARI: Bu işlem geri alınamaz!\n";
echo "Devam etmeden önce veritabanı yedeği aldığınızdan emin olun.\n\n";

if (php_sapi_name() === 'cli') {
    echo "Devam etmek için 'EVET' yazın: ";
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);
    
    if ($line !== 'EVET') {
        die("İşlem iptal edildi.\n");
    }
}

echo "\n=== MIGRATION BAŞLIYOR ===\n\n";

try {
    $db->beginTransaction();
    
    // =====================================================
    // PHASE 1: YENİ SÜTUNLAR EKLE
    // =====================================================
    echo "PHASE 1: Yeni sütunlar ekleniyor...\n";
    
    // fatura_detaylari
    echo "  - fatura_detaylari tablosu...\n";
    $db->exec("ALTER TABLE fatura_detaylari
        ADD COLUMN IF NOT EXISTS birim_fiyat_kurus BIGINT AFTER birim_fiyat,
        ADD COLUMN IF NOT EXISTS toplam_tutar_kurus BIGINT AFTER toplam_tutar,
        ADD COLUMN IF NOT EXISTS kdv_tutari_kurus BIGINT AFTER kdv_tutari,
        ADD COLUMN IF NOT EXISTS satir_indirimi BIGINT DEFAULT 0,
        ADD COLUMN IF NOT EXISTS dagitilan_indirim BIGINT DEFAULT 0,
        ADD COLUMN IF NOT EXISTS kdv_telafi_tutari INT DEFAULT 0,
        ADD COLUMN IF NOT EXISTS kdv_orani_basis SMALLINT AFTER kdv_orani");
    
    // borclar
    echo "  - borclar tablosu...\n";
    $db->exec("ALTER TABLE borclar
        ADD COLUMN IF NOT EXISTS tutar_kurus BIGINT AFTER tutar,
        ADD COLUMN IF NOT EXISTS toplam_kdv_kurus BIGINT AFTER toplam_kdv,
        ADD COLUMN IF NOT EXISTS indirim_tutari_kurus BIGINT AFTER indirim_tutari,
        ADD COLUMN IF NOT EXISTS odenen_tutar_kurus BIGINT AFTER odenen_tutar,
        ADD COLUMN IF NOT EXISTS tax_snapshot LONGTEXT COMMENT 'KDV hesaplama audit trail (JSON)'");
    
    // urunler
    echo "  - urunler tablosu...\n";
    $db->exec("ALTER TABLE urunler
        ADD COLUMN IF NOT EXISTS alis_fiyati_kurus BIGINT AFTER alis_fiyati,
        ADD COLUMN IF NOT EXISTS satis_fiyati_kurus BIGINT AFTER satis_fiyati,
        ADD COLUMN IF NOT EXISTS alis_fiyati2_kurus BIGINT AFTER alis_fiyati2,
        ADD COLUMN IF NOT EXISTS satis_fiyati2_kurus BIGINT AFTER satis_fiyati2,
        ADD COLUMN IF NOT EXISTS kdv_orani_basis SMALLINT AFTER kdv_orani,
        ADD COLUMN IF NOT EXISTS kdv_orani2_basis SMALLINT AFTER kdv_orani2");
    
    // cariler
    echo "  - cariler tablosu...\n";
    $db->exec("ALTER TABLE cariler
        ADD COLUMN IF NOT EXISTS toplam_borc_kurus BIGINT AFTER toplam_borc,
        ADD COLUMN IF NOT EXISTS toplam_alacak_kurus BIGINT AFTER toplam_alacak,
        ADD COLUMN IF NOT EXISTS bakiye_kurus BIGINT AFTER bakiye");
    
    // hesaplar
    echo "  - hesaplar tablosu...\n";
    $db->exec("ALTER TABLE hesaplar
        ADD COLUMN IF NOT EXISTS bakiye_kurus BIGINT AFTER bakiye,
        ADD COLUMN IF NOT EXISTS baslangic_bakiye_kurus BIGINT AFTER baslangic_bakiye");
    
    // hesap_hareketleri
    echo "  - hesap_hareketleri tablosu...\n";
    $db->exec("ALTER TABLE hesap_hareketleri
        ADD COLUMN IF NOT EXISTS tutar_kurus BIGINT AFTER tutar");
    
    // teklifler
    echo "  - teklifler tablosu...\n";
    $db->exec("ALTER TABLE teklifler
        ADD COLUMN IF NOT EXISTS tutar_kurus BIGINT AFTER tutar,
        ADD COLUMN IF NOT EXISTS toplam_kdv_kurus BIGINT AFTER toplam_kdv,
        ADD COLUMN IF NOT EXISTS indirim_tutari_kurus BIGINT AFTER indirim_tutari");
    
    // teklif_detaylari
    echo "  - teklif_detaylari tablosu...\n";
    $db->exec("ALTER TABLE teklif_detaylari
        ADD COLUMN IF NOT EXISTS birim_fiyat_kurus BIGINT AFTER birim_fiyat,
        ADD COLUMN IF NOT EXISTS kdv_tutari_kurus BIGINT AFTER kdv_tutari,
        ADD COLUMN IF NOT EXISTS toplam_tutar_kurus BIGINT AFTER toplam_tutar,
        ADD COLUMN IF NOT EXISTS kdv_orani_basis SMALLINT AFTER kdv_orani");
    
    echo "  ✓ Tüm yeni sütunlar eklendi.\n\n";
    
    // =====================================================
    // PHASE 2: VERİLERİ DÖNÜŞTÜR
    // =====================================================
    echo "PHASE 2: Veriler dönüştürülüyor...\n";
    
    // fatura_detaylari
    echo "  - fatura_detaylari kayıtları...\n";
    $stmt = $db->query("SELECT id, birim_fiyat, toplam_tutar, kdv_tutari, kdv_orani 
                        FROM fatura_detaylari 
                        WHERE birim_fiyat_kurus IS NULL OR birim_fiyat_kurus = 0");
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $birim_kurus = MoneyHelper::toKurus($row['birim_fiyat']);
        $toplam_kurus = MoneyHelper::toKurus($row['toplam_tutar']);
        $kdv_kurus = MoneyHelper::toKurus($row['kdv_tutari']);
        $kdv_basis = TaxEngine::percentToBasis($row['kdv_orani']);
        
        $update = $db->prepare("UPDATE fatura_detaylari 
            SET birim_fiyat_kurus = ?, 
                toplam_tutar_kurus = ?, 
                kdv_tutari_kurus = ?,
                kdv_orani_basis = ?
            WHERE id = ?");
        $update->execute([$birim_kurus, $toplam_kurus, $kdv_kurus, $kdv_basis, $row['id']]);
        
        $count++;
        if ($count % 100 == 0) {
            echo "    $count kayıt işlendi...\n";
        }
    }
    echo "    ✓ Toplam $count fatura detayı dönüştürüldü.\n\n";
    
    // borclar
    echo "  - borclar kayıtları...\n";
    $db->exec("UPDATE borclar SET
        tutar_kurus = ROUND(tutar * 100),
        toplam_kdv_kurus = ROUND(toplam_kdv * 100),
        indirim_tutari_kurus = ROUND(indirim_tutari * 100),
        odenen_tutar_kurus = ROUND(odenen_tutar * 100)
        WHERE tutar_kurus IS NULL OR tutar_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // urunler
    echo "  - urunler kayıtları...\n";
    $db->exec("UPDATE urunler SET
        alis_fiyati_kurus = ROUND(alis_fiyati * 100),
        satis_fiyati_kurus = ROUND(satis_fiyati * 100),
        alis_fiyati2_kurus = ROUND(alis_fiyati2 * 100),
        satis_fiyati2_kurus = ROUND(satis_fiyati2 * 100),
        kdv_orani_basis = ROUND(kdv_orani * 100),
        kdv_orani2_basis = ROUND(kdv_orani2 * 100)
        WHERE alis_fiyati_kurus IS NULL OR alis_fiyati_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // cariler
    echo "  - cariler kayıtları...\n";
    $db->exec("UPDATE cariler SET
        toplam_borc_kurus = ROUND(toplam_borc * 100),
        toplam_alacak_kurus = ROUND(toplam_alacak * 100),
        bakiye_kurus = ROUND(bakiye * 100)
        WHERE toplam_borc_kurus IS NULL OR toplam_borc_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // hesaplar
    echo "  - hesaplar kayıtları...\n";
    $db->exec("UPDATE hesaplar SET
        bakiye_kurus = ROUND(bakiye * 100),
        baslangic_bakiye_kurus = ROUND(baslangic_bakiye * 100)
        WHERE bakiye_kurus IS NULL OR bakiye_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // hesap_hareketleri
    echo "  - hesap_hareketleri kayıtları...\n";
    $db->exec("UPDATE hesap_hareketleri SET
        tutar_kurus = ROUND(tutar * 100)
        WHERE tutar_kurus IS NULL OR tutar_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // teklifler
    echo "  - teklifler kayıtları...\n";
    $db->exec("UPDATE teklifler SET
        tutar_kurus = ROUND(tutar * 100),
        toplam_kdv_kurus = ROUND(toplam_kdv * 100),
        indirim_tutari_kurus = ROUND(indirim_tutari * 100)
        WHERE tutar_kurus IS NULL OR tutar_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // teklif_detaylari
    echo "  - teklif_detaylari kayıtları...\n";
    $db->exec("UPDATE teklif_detaylari SET
        birim_fiyat_kurus = ROUND(birim_fiyat * 100),
        kdv_tutari_kurus = ROUND(kdv_tutari * 100),
        toplam_tutar_kurus = ROUND(toplam_tutar * 100),
        kdv_orani_basis = ROUND(kdv_orani * 100)
        WHERE birim_fiyat_kurus IS NULL OR birim_fiyat_kurus = 0");
    echo "    ✓ Tamamlandı.\n\n";
    
    // =====================================================
    // PHASE 3: KDV AYARI EKLE
    // =====================================================
    echo "PHASE 3: KDV ayarı ekleniyor...\n";
    
    $db->exec("INSERT INTO ayarlar (ayar_adi, ayar_degeri, aciklama, sirket_id) 
               VALUES ('kdv_hesaplama_tipi', 'kdv_haric', 'KDV hesaplama modu: kdv_haric veya kdv_dahil', 1)
               ON DUPLICATE KEY UPDATE ayar_degeri = ayar_degeri");
    
    echo "  ✓ KDV ayarı eklendi.\n\n";
    
    // =====================================================
    // COMMIT
    // =====================================================
    $db->commit();
    
    echo "\n==========================================================\n";
    echo "  ✓ MIGRATION BAŞARIYLA TAMAMLANDI!\n";
    echo "==========================================================\n\n";
    
    echo "SONRAKİ ADIMLAR:\n";
    echo "1. Sistemi test edin\n";
    echo "2. Her şey çalışıyorsa eski sütunları kaldırın:\n";
    echo "   php remove_old_decimal_columns.php\n\n";
    
    echo "NOT: Eski decimal sütunlar şimdilik korundu.\n";
    echo "     Yeni integer sütunlar (_kurus, _basis) kullanılacak.\n\n";
    
} catch (Exception $e) {
    $db->rollBack();
    echo "\n==========================================================\n";
    echo "  ✗ HATA OLUŞTU!\n";
    echo "==========================================================\n\n";
    echo "Hata: " . $e->getMessage() . "\n";
    echo "Dosya: " . $e->getFile() . "\n";
    echo "Satır: " . $e->getLine() . "\n\n";
    echo "Tüm değişiklikler geri alındı.\n\n";
    exit(1);
}
