<?php
/**
 * Rounding Strategy Enum
 * 
 * Different rounding methods for financial calculations
 * 
 * @package Odjoo
 * @category Finance
 * @version 3.0.0
 */
class RoundingStrategy
{
    /**
     * Round half up (standard)
     * 0.5 → 1
     */
    const HALF_UP = 'HALF_UP';
    
    /**
     * Round half down
     * 0.5 → 0
     */
    const HALF_DOWN = 'HALF_DOWN';
    
    /**
     * Round half even (Banker's rounding)
     * 0.5 → nearest even number
     */
    const HALF_EVEN = 'HALF_EVEN';
    
    /**
     * Round down (floor)
     */
    const DOWN = 'DOWN';
    
    /**
     * Round up (ceiling)
     */
    const UP = 'UP';
    
    /**
     * Apply rounding strategy
     * 
     * @param float $value Value to round
     * @param string $strategy Rounding strategy
     * @return int Rounded value
     */
    public static function apply(float $value, string $strategy = self::HALF_UP): int
    {
        switch ($strategy) {
            case self::HALF_UP:
                return (int) round($value, 0, PHP_ROUND_HALF_UP);
                
            case self::HALF_DOWN:
                return (int) round($value, 0, PHP_ROUND_HALF_DOWN);
                
            case self::HALF_EVEN:
                return (int) round($value, 0, PHP_ROUND_HALF_EVEN);
                
            case self::DOWN:
                return (int) floor($value);
                
            case self::UP:
                return (int) ceil($value);
                
            default:
                throw new InvalidArgumentException("Invalid rounding strategy: {$strategy}");
        }
    }
    
    /**
     * Get default strategy for Turkey
     * 
     * @return string
     */
    public static function getDefault(): string
    {
        return self::HALF_UP;
    }
    
    /**
     * Validate strategy
     * 
     * @param string $strategy
     * @return bool
     */
    public static function isValid(string $strategy): bool
    {
        return in_array($strategy, [
            self::HALF_UP,
            self::HALF_DOWN,
            self::HALF_EVEN,
            self::DOWN,
            self::UP
        ], true);
    }
}
